/****************************************************************************
**
**	File Name   : SV6Chunk.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Thu 20 Mar '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the Chunk class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Chunk.cpp 4     5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Wed 05 Mar '03		1.00	Initial design & coding
**	Mon 10 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global Constants/Variables
	----------------------------------------------------------------------	*/
static const uint32		kBufferSize = MB(3);

static uint8			s_buffer[kBufferSize];
/*	----------------------------------------------------------------------
	Local Function Prototypes
	----------------------------------------------------------------------	*/

/*lint -save -e661 -e668 -e669
**	For some reason, the pointer bashing in ::Read() drives PC-Lint nuts...
*/

/****************************************************************************
**
**	Chunk::Chunk() -- [CTOR]
**
*/
Chunk::Chunk() :	m_bDirty(false),
					m_nIndex(-1),
					m_dwDataLen(0),
					m_pRawData(NULL)
{
MEM_ZERO(&m_info, sizeof(SV6ChunkInfo));
}	/* end of Chunk::Chunk() */
/****************************************************************************
**
**	Chunk::~Chunk() -- [DTOR]
**
*/
Chunk::~Chunk()
{
delete [] m_pRawData;
m_pRawData = NULL;
}	/* end of Chunk::~Chunk() */
/****************************************************************************
**
**	Chunk::allocateData() -- allocates a chunk of memory 
**
*/
void
Chunk::allocateData(
	uint32	dwSize)				// bytes to allocate (or zero to free)
{
if (GOOD_PTR(m_pRawData))
	delete [] m_pRawData;

m_pRawData = NULL;
m_dwDataLen = 0;

if (dwSize > 0)
	{
	m_pRawData = new uint8[dwSize];
	if (NULL_PTR(m_pRawData))
		{
		Display(true,
				"Could not allocate %s byte buffer!\n",
				StrNum(dwSize));
		throw uint32( ERROR_NOT_ENOUGH_MEMORY );
		}

	m_dwDataLen = dwSize;
	m_bDirty = false;
	}
}	/* end of Chunk::allocateData() */
/****************************************************************************
**
**	Chunk::compress() -- compresses a chunk of RCT data into s_buffer
**
*/
uint32							// size of compressed data, or INVALID_FILE_SIZE on error
Chunk::compress(
	uint8 &		xType,			// type of compression (0x00...0x03)
	DataPtr		pData,			// data to compress
	uint32		dwDataLen)		// length of data (in bytes)
{
uint32	dwBytes;
uint8 *	pBuffer = s_buffer;

if (NULL_PTR(pData) || dwDataLen == 0)
	throw uint32( ERROR_INVALID_PARAMETER );

dwBytes = dwDataLen + 1;
//
//	Now see if we EXPANDED the data by compressing it!
//
if (dwBytes >= dwDataLen)
	{
	xType = 0x00;
	dwBytes = dwDataLen;
	MEM_COPY(pBuffer, pData, dwDataLen);

	Display(false,
			"Chunk #%d: no compression (%s bytes).\n",
			m_nIndex, StrNum(dwDataLen));
	}
else
	{
	Display(false,
			"Chunk #%d: compressed data %s bytes to %s file bytes\n",
			m_nIndex, StrNum(dwDataLen), StrNum(dwBytes));
	}

return dwBytes; 
}	/* end of Chunk::compress() */
/****************************************************************************
**
**	Chunk::decompress() -- decompresses a chunk of RCT data into s_buffer
**
*/
uint32							// NO_ERROR on success, false otherwise
Chunk::decompress(
	uint8		xType,			// decompression type (0x00...0x03)
	DataPtr		pData,			// ptr to compressed data
	uint32		dwDataLen)		// length of compressed data in bytes
{
uint32	dwSize = 0;
uint8 *	pBuffer = s_buffer;

switch (xType)
	{
	case 0x00:	// no compression
		MEM_COPY(pBuffer, pData, dwDataLen);
		dwSize = dwDataLen;
		break;

	case 0x01:	// RLE compression
		{
		uint8	xVal;
		int32	nCount;
		uint32	dwNext = 0;

		do
			{
			xVal = pData[dwNext++];
			//
			//	Test the MSB.  If it is 1, then the next byte is copied (1 - xVal)
			//	times.  If it is 0, then the next xVal bytes are copied as is.
			//
			if (xVal & 0x80)
				{
				nCount = 1 - int8(xVal);	// treat as a signed value
				if (nCount <= 0 || nCount >= 0x80 || dwNext >= dwDataLen)
					return ERROR_INVALID_DATA;		// not enough src data

				if ((dwSize + nCount) > kBufferSize)
					return ERROR_BUFFER_OVERFLOW;	// not enough dst room

				xVal = pData[dwNext++];
				MEM_FILL(&pBuffer[dwSize], UINT(nCount), xVal);
				}
			else
				{
				nCount = 1 + int8(xVal);	// treat as a signed value
				if (nCount <= 0 || (dwNext + nCount) > dwDataLen)
					return ERROR_INVALID_DATA;		// not enough src data

				if ((dwSize + nCount) > kBufferSize)
					return ERROR_BUFFER_OVERFLOW;	// not enough dst room

				MEM_COPY(&pBuffer[dwSize], &pData[dwNext], UINT(nCount));

				dwNext += nCount;
				}

			dwSize += nCount;
			}
		while (dwNext < dwDataLen);
		}
		break;

	case 0x02:
		{
		uint8		xVal;
		uint32		dwErr,
					dwNext = 0;

		dwErr = decompress(0x01, pData, dwDataLen);	// do the normal RLE decompression
		if (dwErr != NO_ERROR)
			return dwErr;

		pData		= m_pRawData;
		dwDataLen	= m_dwDataLen;

		do
			{
			xVal = pData[dwNext++];
			if (xVal == 0xFF)
				{
				//
				//	Copy the next byte directly into the output data stream
				//
				if (dwNext >= dwDataLen)
					return ERROR_INVALID_DATA;		// not enough src data

				if (dwSize >= kBufferSize)
					return ERROR_BUFFER_OVERFLOW;	// not enough dst room

				pBuffer[dwSize++] = pData[dwNext++];	//lint !e661
				}
			else
				{
				uint32	dwLen = (xVal & 0x07) + 1;
				int32	nOffset = int32((xVal & 0xF8) >> 3) - 32L;

				if ((dwSize + nOffset) >= dwSize)
					return ERROR_INVALID_DATA;		// not enough src data

				if ((dwSize + dwLen) > kBufferSize)
					return ERROR_BUFFER_OVERFLOW;	// not enough dst room

				MEM_COPY(&pBuffer[dwSize], &pBuffer[dwSize + nOffset], dwLen);

				dwSize += dwLen;
				}
			}
		while (dwNext < dwDataLen);
		}
		break;

	case 0x03:
		{
		int32	nRotateBy = 1;

		MEM_COPY(pBuffer, pData, dwDataLen);
		dwSize = dwDataLen;

		for (uint32 dwx = 0; dwx < dwSize; ++dwx, ++pBuffer)
			{
			if (*pBuffer)	// don't bother to rotate if zero
				*pBuffer = ROTR(*pBuffer, nRotateBy, 8);

			if ((nRotateBy += 2)  > 7)
				nRotateBy = 1;
			}
		}
		break;

	default:
		Display(true,
				"Chunk #%d: invalid compression type (0x%02X)\n",
				m_nIndex, xType);
		return ERROR_INVALID_DATA;
	}
//
//	Copy the uncompressed data from the temporary buffer into our buffer
//
allocateData(dwSize);
PANIC_IF_NULL(m_pRawData);
MEM_COPY(m_pRawData, s_buffer, dwSize);

return NO_ERROR;
}	/* end of Chunk::decompress() */
/****************************************************************************
**
**	Chunk::readChunk() -- reads in a chunk of RCT data
**
*/
bool							// true if read successfully, false otherwise
Chunk::readChunk(
	DataPtr	pData,		// ptr to start of chunk
	uint32			dwDataLen,	// data bytes available
	uint32 &		dwRead)		// [out] size of chunk
{
dwRead = 0;

if (dwDataLen < sizeof(SV6ChunkInfo))
	{
	Display(true, 
			"Chunk #%d: Not enough header data.\n",
			m_nIndex);
	return false;
	}

MEM_COPY(&m_info, pData, sizeof(SV6ChunkInfo));
dwRead += sizeof(SV6ChunkInfo);

if (m_info.dwChunkSize == 0 || m_info.dwChunkSize > dwDataLen)
	{
	Display(true,
			"Chunk #%d: invalid data length (%s bytes).\n",
			m_nIndex, StrNum(m_info.dwChunkSize));
	return false;
	}
//
//	Now decompress the data
//
uint32	dwErr;

dwErr = decompress(m_info.xCompression, &pData[dwRead], m_info.dwChunkSize);
if (dwErr != NO_ERROR)
	{
	Display(false,
			"Chunk #%d: decompression failure (error #%lu)\n",
			m_nIndex, dwErr);
	return false;
	}

dwRead += m_info.dwChunkSize;

Display(false,
		"Chunk #%d: decompressed %s file bytes to %s data bytes\n",
		m_nIndex, StrNum(m_info.dwChunkSize), StrNum(m_dwDataLen));

return true;
}	/* end of Chunk::readChunk() */
/****************************************************************************
**
**	Chunk::writeChunk() -- writes out a chunk of RCT data
**
*/
bool							// true if succesful, false otherwise
Chunk::writeChunk(
	FILE *		pfOut,			// ptr to output file
	uint32 &	dwBytes,		// running count of bytes written
	uint32 &	dwCRC)			// running CRC value (simple accumulator)
{
uint8 *		pData;
uint32		dwActual,
			dwWanted;
//
//	Write out the chunk header
//
m_info.xCompression	= 0x00;
m_info.dwChunkSize	= compress(m_info.xCompression, m_pRawData, m_dwDataLen);

pData = reinterpret_cast<uint8 *>(&m_info); 
dwWanted = sizeof(SV6ChunkInfo);
dwActual = uint32( fwrite(pData, 1, dwWanted, pfOut) );
if (dwWanted != dwActual)
	return false;

dwBytes += dwActual;
for (dwWanted = 0; dwWanted < dwActual; ++dwWanted)
	dwCRC += *pData++;
//
//	And the data
//
pData = s_buffer; 
dwWanted = m_info.dwChunkSize;
dwActual = uint32( fwrite(pData, 1, dwWanted, pfOut) );
if (dwWanted != dwActual)
	return false;

dwBytes += dwActual;
for (dwWanted = 0; dwWanted < dwActual; ++dwWanted)
	dwCRC += *pData++;

Display(false,
		"Chunk #%d: wrote out %s bytes\n",
		m_nIndex, StrNum(sizeof(SV6ChunkInfo) + dwActual));

return true;
}	/* end of Chunk::writeChunk() */
/****************************************************************************
**
**	End of SV6Chunk.cpp
**
****************************************************************************/
