/****************************************************************************
**
**	File Name   : SV6Data.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Tue 27 May '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the CSV6Data class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Data.cpp 10    5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Thu 06 Mar '03		1.00	Initial design & coding
**	Thu 13 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global (external) Constants/Variables
	----------------------------------------------------------------------	*/

/*	----------------------------------------------------------------------
	Local (static) Constants/Variables
	----------------------------------------------------------------------	*/
#define	PRICE_STR(n)	((n) ? StrCash((n), 10) : "free")
/*	----------------------------------------------------------------------
	Local (static) Functions
	----------------------------------------------------------------------	*/

/****************************************************************************
**
**	CSV6Data::CSV6Data() -- [CTOR]
**
*/
CSV6Data::CSV6Data() :	m_pData(NULL),
						m_dwRides(0),
						m_dwSprites(0)
{
ASSERT( sizeof(SV6Ride) == 0x260 );
ASSERT( sizeof(SV6Sprite) == 0x100 );
}	/* end of CSV6Data::CSV6Data() */
/****************************************************************************
**
**	CSV6Data::~CSV6Data() -- [DTOR]
**
*/
CSV6Data::~CSV6Data()
{
m_pData = NULL;	// don't free it, because it really points to m_pRawData
}	/* end of CSV6Data::~CSV6Data() */
/****************************************************************************
**
**	CSV6Data::Apply() -- apply changes to this chunk
**
*/
bool							// true if data changed, false otherwise
CSV6Data::Apply(
	Changes const &		ch)		// contains all changes to be applied
{
bool	bNewCRC = ch.bFix;

PANIC_IF_NULL(m_pData);

if (ch.bPeepsBuff && (getStatusBit(6) || !getStatusBit(9)))
	{
	m_bDirty = true;

	(void) setStatusBit(6, false);
	(void) setStatusBit(9, true);
	}

if (ch.bMakeFree && !getStatusBit(11))
	{
	m_bDirty = true;
	
	m_pData->wParkEntryFee = 0;		// park entry is free
	(void) setStatusBit(0, true);	// and make sure park is open
	(void) setStatusBit(11, true);	// and make everything free

	Display(true, "Everything is now free.\n");
	}

if (ch.bMakeMoney)
	{
	if (getStatusBit(7) &&
		setStatusBit(7, false))
		{
		m_bDirty = true;
		Display(true, "Advertising can now be commissioned.\n");
		}

	if (getStatusBit(11) &&
		setStatusBit(11, false))	// money matters now...
		{
		m_bDirty = true;
		Display(true, "Everything has a price.\n");
		}

	if (getStatusBit(13) &&
		setStatusBit(13, false))
		{
		m_bDirty = true;
		Display(true, "Park Entrance fee can now be changed.\n");
		}
	}

if (ch.wParkRating && 
	setParkRating(ch.wParkRating - 1))
	{
	m_bDirty = true;
	Display(true,
			"Park rating changed to %s\n",
			StrNum(GetParkRating()));
	}

if (ch.dwCash && setCash(ch.dwCash - 1))
	{
	m_bDirty = bNewCRC = true;
	Display(true,
			"Cash amount changed to %s\n",
			StrCash(GetCash(), 10));
	}

if (ch.wIntRate && setIntRate(ch.wIntRate - 1))
	{
	m_bDirty = true;
	Display(true,
			"Interest rate changed to %s%%\n",
			StrNum(GetIntRate()));
	}
//
//	Set the max. loan amount FIRST so that any changes to the loan amount
//	will respect the new maximum.
//
if (ch.dwMaxLoan && setMaxLoan(ch.dwMaxLoan - 1))
	{
	m_bDirty = bNewCRC = true;
	Display(true,
			"Maximum loan amount changed to %s\n",
			StrCash(GetMaxLoan(), 10));
	}
//
//	Set the loan amount, or fix it if necessary
//
uint32	dwLoan = ch.dwLoan;		// use whatever user specified on command line

if (ch.bFix &&					// if we're fixing stuff...
	!dwLoan &&					// ...and the user didn't set a loan amount
	GetLoan() > GetMaxLoan())	// ...and the current value is too big
	{
	dwLoan = GetMaxLoan() + 1;	// ...then set the loan to the max amount
	Display(true, "FIXED: loan amount exceeds maximum loan\n");
	}

if (dwLoan && setLoan(dwLoan - 1))
	{
	m_bDirty = bNewCRC = true;
	Display(true,
			"Loan amount changed to %s\n",
			StrCash(GetLoan(), 10));
	}
//
//	If we're giving peeps money, make sure that peeps arriving get the same
//	benefit.  Unlike the per-peep value, this is in whole dollars and must be
//	at least $10 (otherwise peeps could come in with negative amounts).
//
if (ch.dwPeepCash > 10)
	{
	m_bDirty = true;
	m_pData->wPeepCash = uint16(ch.dwPeepCash) - 1;
	Display(true,
			"Arriving peeps will now carry between %s and %s.\n",
			StrCash(m_pData->wPeepCash - 10),
			StrCash(m_pData->wPeepCash + 20));
	}
//
//	If we changed any of the financial data, we have to update the CRCs.
//	Technically, CRC#2 is only dependent on the cash, but it's just as easy
//	to recompute them all.
//
if (bNewCRC)
	{
	m_bDirty = true;
	m_pData->dwCashCRC1 = computeCRC1();
	m_pData->dwCashCRC2 = computeCRC2();
	Display(false,
			"Internal CRC #1 is 0x%08lX\n"
			"Internal CRC #2 is 0x%08lX\n",
			m_pData->dwCashCRC1, m_pData->dwCashCRC2);
	}
/*
**	Apply changes to the rides. 
*/
CSV6Ride	ride;
uint32		dwx,
			dwRides;

for (dwx = dwRides = 0; dwx < kRideCount; ++dwx)
	{
	if (ride.Set(&m_pData->rides[dwx]) &&
		ride.Apply(ch, dwx))
		{
		dwRides++;
		}
	}

if (dwRides > 0)
	{
	m_bDirty = true;
	Display(true,
			"Modified %s rides\n",
			StrNum(dwRides));
	}
/*
**	Apply changes to the sprites.  Although we dump the sprites in chains, we
**	make the changes in sequential order, because changing the sprite data
**	might clobber the wNextSprite link.
*/
CSV6Sprite	sprite;
uint32		dwSprites;

for (dwx = dwSprites = 0; dwx < kSpriteCount; ++dwx)
	{
	if (sprite.Set(&m_pData->sprites[dwx]) &&
		sprite.Apply(ch, dwx))
		{
		dwSprites++;
		}
	}

if (dwSprites > 0)
	{
	m_bDirty = true;
	Display(true,
			"Modified %s sprites.\n",
			StrNum(dwSprites));
	}

if (ch.bPeepsGone)
	{
	//
	//	Reset the peep count to zero.  I'm not sure why yet, but wPeepCount
	//	does NOT match the # of peep sprites being deleted; this may have to
	//	do with peeps that are on the map, but not yet in the park.
	//
	m_pData->wPeepCount = 0;
	//
	//	Walk the sprite and perform the following actions:  (1) if we find a 
	//	peep, nuke the sprite info (fill it with 0xFF).  (2) if we find any
	//	other type of person (staff) then update the links.
	//
	uint16		wPrev = 0xFFFF;

	for (dwx = dwSprites = 0; dwx < kSpriteCount; ++dwx)
		{
		if (sprite.Set(&m_pData->sprites[dwx]) &&
			sprite.GetType() == uint8(CSV6Sprite::ST_PERSON))
			{
			if (sprite.IsPeep())
				{	// for peeps, nuke the sprite information
				dwSprites++;
				m_pData->wPersonCount--;
				m_pData->wSpritesAvailable++;
				MEM_FILL(&m_pData->sprites[dwx], sizeof(SV6Sprite), 0xFF);
				continue;	// done with this sprite
				}
			//
			//	OK...it's a person, but not a peep (includes guests not yet
			//	in the park?)
			//
			if (wPrev == 0xFFFF)	// true if first non-peep person
				{
				m_pData->wPersonSprite = wPrev = uint16(dwx & 0xFFFF);

				m_pData->sprites[dwx].wPrevSprite = 0xFFFF;
				m_pData->sprites[dwx].wNextSprite = 0xFFFF;
				}
			else	// Nth non-peep person
				{
				uint16	wHere = uint16(dwx & 0xFFFF);

				ASSERT(wPrev >= 0 && wPrev < kSpriteCount);
				m_pData->sprites[wPrev].wNextSprite = wHere;

				m_pData->sprites[dwx].wPrevSprite = wPrev;
				m_pData->sprites[dwx].wNextSprite = 0xFFFF;
				wPrev = wHere;
				}
			}
		}

	if (dwSprites > 0)
		{
		m_bDirty = true;
		Display(true,
				"Made %s peeps disappear.\n",
				StrNum(dwSprites));
		}
	}

if (ch.bMakeTidy)
	{
	//
	//	Walk the sprite and if we find any trash, nuke the sprite info (fill 
	//	it with 0xFF.
	//
	for (dwx = dwSprites = 0; dwx < kSpriteCount; ++dwx)
		{
		if (sprite.Set(&m_pData->sprites[dwx]) &&
			sprite.GetType() == uint8(CSV6Sprite::ST_TRASH))
			{
			dwSprites++;
			MEM_FILL(&m_pData->sprites[dwx], sizeof(SV6Sprite), 0xFF);
			}
		}

	ASSERT( dwSprites == m_pData->wTrashCount );

	if (dwSprites > 0)
		{
		m_bDirty = true;
		Display(true,
				"Picked up %s pieces of trash.\n",
				StrNum(dwSprites));
		}

	m_pData->wSpritesAvailable += m_pData->wTrashCount;
	m_pData->wTrashCount = 0;
	m_pData->wTrashSprite = 0xFFFF;
	}

ASSERT( verifyChain(m_pData->wVehicleSprite, m_pData->wVehicleCount, CSV6Sprite::ST_VEHICLE) );
ASSERT( verifyChain(m_pData->wPersonSprite, m_pData->wPersonCount, CSV6Sprite::ST_PERSON) );
ASSERT( verifyChain(m_pData->wFlyerSprite, m_pData->wFlyerCount, CSV6Sprite::ST_FLYER) );
ASSERT( verifyChain(m_pData->wTrashSprite, m_pData->wTrashCount, CSV6Sprite::ST_TRASH) );
ASSERT( verifyChain(m_pData->wOversizedSprite, m_pData->wOversizedCount, CSV6Sprite::ST_VEHICLE) );

return m_bDirty;
}	/* end of CSV6Data::Apply() */
/****************************************************************************
**
**	CSV6Data::Dump() -- dumps all generic information
**
*/
void
CSV6Data::Dump(
	Options const &	opt) const	
{
PANIC_IF_NULL(m_pData);

Display(opt.bDumpInfo,
		"Scenario Name     : %s\n"
		"Scenario Details  : %s\n"
		"Park Status       : %15s\n",
		GetScenarioName(), 
		GetScenarioDetails(),
		(getStatusBit(0) ? "Open" : "Closed"));

if (!getStatusBit(11))	// don't bother if money is irrelevant
	{
	Display(opt.bDumpInfo,
			"Park Entrance Fee : %15s %s\n"
			"Cash              : %15s\n"
			"Current loan      : %15s\n"
			"Maximum loan      : %15s\n"
			"Interest Rate     : %14s%%\n",
			PRICE_STR(m_pData->wParkEntryFee),
			(getStatusBit(13) ? "(fixed)" : kStrEmpty),
			StrCash(GetCash(), 10),
			StrCash(GetLoan(), 10), 
			StrCash(GetMaxLoan(), 10), 
			StrNum(GetIntRate()));
	}

Display(opt.bDumpInfo,
		"Park Rating       : %15d\n"
		"Peep Count        : %15s\n"
		"Map Dimensions    :       %3d x %3d\n",
		GetParkRating(),
		StrNum(GetPeepCount()),
		GetMapXDim(), GetMapYDim());

#if	0
Display(opt.bDumpInfo, "Status Bits       : ");
for (uint32 dwBit = 0; dwBit < 32; ++dwBit)
	{
	Display(opt.bDumpInfo, (getStatusBit(dwBit) ? "1" : "0"));
	if (((dwBit + 1) % 8) == 0)			// every 8th bit...
		Display(opt.bDumpInfo, " ");	// put in a space for clarity
	}
#endif

Display(opt.bDumpInfo, kStrNewLn);
/*
**	Now dump the rides
*/
DisplayBar(opt.bDumpRides, '=', opt.nOutputWidth);

if (m_dwRides > 0)
	{
	CSV6Ride	ride;

	Display(opt.bDumpRides,
			"Ride    Type                            Status  Built\n");

	for (uint32 uRide = 0; uRide < kRideCount; ++uRide)
		if (ride.Set(&m_pData->rides[uRide]))
			ride.Dump(opt, uRide);

	Display(opt.bDumpRides,
			"\n%s rides in use\n",
			StrNum(m_dwRides));
	}
else
	Display(opt.bDumpRides, "No rides in use.\n");

DisplayBar(opt.bDumpRides, '=', opt.nOutputWidth);
/*
**	Now dump the sprites
*/
DisplayBar(opt.bDumpSprites, '=', opt.nOutputWidth);

if (m_dwSprites > 0)
	{
	dumpChain(opt, m_pData->wVehicleSprite, m_pData->wVehicleCount,
				"Vehicle Sprites");
	dumpChain(opt, m_pData->wPersonSprite, m_pData->wPersonCount,
				"Guests/Staff Sprites");
	dumpChain(opt, m_pData->wFlyerSprite, m_pData->wFlyerCount,
				"Flyer Sprites");
	dumpChain(opt, m_pData->wTrashSprite, m_pData->wTrashCount,
				"Trash/Vomit Sprites");
	dumpChain(opt, m_pData->wOversizedSprite, m_pData->wOversizedCount,
				"Oversized Vehicle Sprites");

	Display(opt.bDumpSprites,
			"TOTAL: %s sprites in use.\n",
			StrNum(m_dwSprites));
	}
else
	{
	Display(opt.bDumpSprites, "No sprites in use.\n");
	}

DisplayBar(opt.bDumpSprites, '=', opt.nOutputWidth);
}	/* end of CSV6Data::Dump() */
/****************************************************************************
**
**	CSV6Data::Read() -- reads in a chunk of RCT data
**
*/
bool							// true if read successfully, false otherwise
CSV6Data::Read(
	int32		nIndex,			// chunk #
	DataPtr		pData,			// ptr to start of chunk
	uint32		dwDataLen,		// data bytes available
	uint32 &	dwRead)			// [out] size of chunk
{
m_nIndex = nIndex;		ASSERT(nIndex == 5);
	
if (!readChunk(pData, dwDataLen, dwRead))
	return false;

if (m_dwDataLen != 0x2E8570)	// 3048816
	{
	Display(true,
			"Chunk #%ld: invalid data length (%s bytes).\n",
			m_nIndex, StrNum(m_dwDataLen));
	return false;
	}
/*
**	Now validate the data
*/
m_pData = reinterpret_cast<SV6DataPtr>( m_pRawData );

if (NULL_PTR(m_pData))
	{
	Display(true,
			"Chunk #%ld: invalid data.\n",
			m_nIndex);
	return false;
	}
//
//	Check the Cash CRC
//
if (computeCRC1() == m_pData->dwCashCRC1)
	{
	Display(false,
			"Chunk #%ld: internal CRC #1 OK\n",
			m_nIndex);
	}
else
	{
	Display(true,
			"WARNING: internal checksum #1 failed.\n",
			m_nIndex);
	}

if (computeCRC2() == m_pData->dwCashCRC2)
	{
	Display(false,
			"Chunk #%ld: internal CRC #2 OK\n",
			m_nIndex);
	}
else
	{
	Display(true,
			"WARNING: Chunk #%ld has a bad internal checksum #2.\n",
			m_nIndex);
	}
//
//	Check the rides
//
CSV6Ride	ride;
uint32		dwx,
			dwEmpty;

for (dwx = dwEmpty = 0; dwx < kRideCount; ++dwx)
	{
	if (ride.Set(&m_pData->rides[dwx]))
		{
		if (ride.IsEmpty())
			dwEmpty++;
		}
	else
		{
		Display(true,
				"Chunk #%ld: ride #%s has invalid data\n",
				m_nIndex, StrNum(dwx));
		return false;
		}
	}

m_dwRides = kRideCount - dwEmpty;
//
//	Check the sprites
//
CSV6Sprite	sprite;

for (dwx = dwEmpty = 0; dwx < kSpriteCount; ++dwx)
	{
	if (!sprite.Set(&m_pData->sprites[dwx]))
		{
		Display(true,
				"Chunk #%ld: sprite 0x%04X has invalid data\n",
				m_nIndex, dwx);
		return false;
		}

	if (sprite.IsEmpty())
		dwEmpty++;
	}

m_dwSprites = kSpriteCount - dwEmpty;
if (dwEmpty != m_pData->wSpritesAvailable)
	{
	Display(true,
			"WARNING:  Chunk #%ld has an invalid \"available\" sprite count.\n",
			m_nIndex);
	}
else if (m_dwSprites !=  uint32(m_pData->wVehicleCount + 
								m_pData->wPersonCount + 
								m_pData->wFlyerCount + 
								m_pData->wTrashCount + 
								m_pData->wOversizedCount))
	{
	Display(true,
			"WARNING:  Chunk #%ld has an invalid \"in-use\" sprite count.\n",
			m_nIndex);
	}
else
	{
	Display(false,
			"Chunk #%ld: %s sprites in use, %s available.\n",
			m_nIndex, StrNum(m_dwSprites), StrNum(dwEmpty));
	}

if (m_dwSprites < 1)
	return true;	// nothing to do if there aren't any sprites to check
//
//	Walk the sprite chains.  The "Oversized" chain is actually made up of
//	vehicle sprites.
//
if (!verifyChain(m_pData->wVehicleSprite, m_pData->wVehicleCount, CSV6Sprite::ST_VEHICLE) ||
	!verifyChain(m_pData->wPersonSprite, m_pData->wPersonCount, CSV6Sprite::ST_PERSON) ||
	!verifyChain(m_pData->wFlyerSprite, m_pData->wFlyerCount, CSV6Sprite::ST_FLYER) ||
	!verifyChain(m_pData->wTrashSprite, m_pData->wTrashCount, CSV6Sprite::ST_TRASH) ||
	!verifyChain(m_pData->wOversizedSprite, m_pData->wOversizedCount, CSV6Sprite::ST_VEHICLE))
	{
	return false;
	}

return true;
}	/* end of CSV6Data::Read() */
/****************************************************************************
**
**	CSV6Data::computeCRC1() -- compute the 1st "Cash CRC" 
**		0x70093A minus cash, ror 5, minus loan, ror 7, add maxloan, ror 3
**
*/
uint32							// new 32-bit CRC
CSV6Data::computeCRC1(void) const
{
uint32	dwCRC = 0x70093A - m_pData->dwCash;

dwCRC = ROTR(dwCRC, 5, 32) - m_pData->dwLoan;
dwCRC = ROTR(dwCRC, 7, 32) + m_pData->dwMaxLoan;

return ROTR(dwCRC, 3, 32);
}	/* end of CSV6Data::computeCRC1() */
/****************************************************************************
**
**	CSV6Data::computeCRC2() -- compute the 2nd "Cash CRC"
**		cash ror thirteen, xor 0F4EC9621
**
*/
uint32							// new 32-bit CRC
CSV6Data::computeCRC2(void) const
{
uint32	dwCRC = ROTR(m_pData->dwCash, 13, 32);

return dwCRC ^ 0x0F4EC9621;
}	/* end of CSV6Data::computeCRC2() */
/****************************************************************************
**
**	CSV6Data::dumpChain() -- walks a chain of sprites, dumping each sprite
**
*/
void
CSV6Data::dumpChain(
	Options const &	opt,		// options
	uint16			wSprite,	// index of first sprite in chain
	uint32			dwCount,	// expected # of sprites chain
	TextPtr			pstrTitle) const
{
CSV6Sprite	sprite;

Display(opt.bDumpSprites, 
		"%s (%s sprites)\n"
		"Sprite Type\n", 
		STR_SAFE(pstrTitle), StrNum(dwCount));

for (uint32 dwx = 0; wSprite < kSpriteCount; ++dwx)
	{
	if (sprite.Set(&m_pData->sprites[wSprite]))
		{
		sprite.Dump(opt, wSprite);
		wSprite = sprite.GetNext();
		}
	else
		wSprite = 0xFFFF;
	//
	//	If the counter (dwCount) exceeds the number of sprites expected, then
	//	we're obviously looping...
	// 
	if (dwx > dwCount)
		throw uint32( ERROR_INVALID_DATA );
	}

DisplayBar(opt.bDumpSprites, '=', opt.nOutputWidth);
}	/* end of CSV6Data::dumpChain() */
/****************************************************************************
**
**	CSV6Data::getStatusBit() -- returns one of the 32 status flags in the 
**		xGameFlags[] array.
**
****************************************************************************/
bool									// value of bit (false if invalid)
CSV6Data::getStatusBit(
	int32		nFlag) const			// bit # to return (0..31)
{
return (GOOD_PTR(m_pData) &&
		(nFlag >= 0 && nFlag < 32) &&
		(m_pData->xGameFlags[nFlag / 8] & (1 << (nFlag & 0x07))));
}	/* end of CSV6Data::getStatusBit() */
/****************************************************************************
**
**	CSV6Data::setStatusBit() -- sets/clears one of the 32 status flags in 
**		the xGameFlags[] array.
**
****************************************************************************/
bool							// true if valid set, false otherwise
CSV6Data::setStatusBit(
	int32	nFlag, 				// bit # to set/clear (0..23)
	bool	bOn)
{
if (GOOD_PTR(m_pData) && (nFlag >= 0 && nFlag < 32))
	{
	if (bOn)
		m_pData->xGameFlags[nFlag / 8] |= (1 << (nFlag & 0x07));
	else
		m_pData->xGameFlags[nFlag / 8] &= ~(1 << (nFlag & 0x07));

	return true;
	}

return false;
}	/* end of CSV6Data::setStatusBit() */
/****************************************************************************
**
**	CSV6Data::setCash() -- set the current cash amount
**
*/
bool							// true if set, false otherwise
CSV6Data::setCash(
	uint32	dwCash)				// desired cash amount (in whole dollars)
{
if (GOOD_PTR(m_pData) &&
	(dwCash >= kMinCash && dwCash <= kMaxCash))	//lint !e568 !e685
	{
	m_pData->dwCash = dwCash * 10;
	return true;
	}

return false;
}	/* end of CSV6Data::setCash() */
/****************************************************************************
**
**	CSV6Data::setIntRate() -- set the current interest rate
**
*/
bool							// true if set, false otherwise
CSV6Data::setIntRate(
	uint8	xIntRate)			// desired interest rate
{
if (GOOD_PTR(m_pData) &&
	(xIntRate >= kMinIntRate && xIntRate <= kMaxIntRate))	//lint !e568 !e685
	{
	m_pData->xInterestRate = xIntRate;
	return true;
	}

return false;
}	/* end of CSV6Data::setIntRate() */
/****************************************************************************
**
**	CSV6Data::setLoan() -- set the current loan amount.  The current maximum
**		loan amount is respected, i.e., you can't set the amount higher than
**		the allowed value.
**
*/
bool							// true if set, false otherwise
CSV6Data::setLoan(
	uint32	dwLoan)				// desired loan amount  (in whole dollars)
{
if (GOOD_PTR(m_pData) &&
	(dwLoan >= kMinLoan && dwLoan <= kMaxLoan))	//lint !e568 !e685
	{
	m_pData->dwLoan = dwLoan * 10;
	//
	//	Don't set the loan beyond the allowed maximum.
	//
	if (m_pData->dwLoan > m_pData->dwMaxLoan)
		m_pData->dwLoan = m_pData->dwMaxLoan;	

	return true;
	}

return false;
}	/* end of CSV6Data::setLoan() */
/****************************************************************************
**
**	CSV6Data::setMaxLoan() -- set the maximum allowed loan amount.  The amount
**		of the current loan is adjusted to make sure that it is stil within
**		range.
**
*/
bool							// true if set, false otherwise
CSV6Data::setMaxLoan(
	uint32	dwMaxLoan)			// desired maximum loan amount  (in whole $)
{
if (GOOD_PTR(m_pData) &&
	(dwMaxLoan >= kMinLoan && dwMaxLoan <= kMaxLoan))	//lint !e568 !e685
	{
	m_pData->dwMaxLoan = dwMaxLoan * 10;
	//
	//	If the max. loan amount drops below the current loan amount, reset 
	//	the current loan to the max. allowed.
	//
	if (m_pData->dwCash > m_pData->dwMaxLoan)
		m_pData->dwCash = m_pData->dwMaxLoan;

	return true;
	}

return false;
}	/* end of CSV6Data::setMaxLoan() */
/****************************************************************************
**
**	CSV6Data::setParkRating() -- set the maximum allowed loan amount
**
*/
bool							// true if set, false otherwise
CSV6Data::setParkRating(
	uint16	wParkRating)		// desired park rating
{
if (GOOD_PTR(m_pData) &&
	(wParkRating >= kMinRating && wParkRating <= kMaxRating))	//lint !e568 !e685
	{
	m_pData->wParkRating = wParkRating;
	return true;
	}

return false;
}	/* end of CSV6Data::setParkRating() */
/****************************************************************************
**
**	CSV6Data::verifyChain() -- walks a chain of sprites, verifying that the
**		chain is correct.
**
*/
bool							// true if chain is valid, false on error
CSV6Data::verifyChain(
	uint16		wSprite,		// index of first sprite in chain
	uint16		wExpected,		// expected count of sprites in chain
	const CSV6Sprite::SpriteType	st) const
{
CSV6Sprite	sprite;
uint32		dwCount;

for (dwCount = 0; wSprite < kSpriteCount; ++dwCount)
	{
	if (!sprite.Set(&m_pData->sprites[wSprite]) || sprite.GetType() != uint8(st))
		{
		Display(true,
				"Chunk #%ld: corrupt sprite 0x%04X in chain type 0x%02lX.\n",
				m_nIndex, wSprite, uint32(st));
		return false;
		}

	wSprite = sprite.GetNext();
	//
	//	If the sprite counter (dwCount) exceeds the total number of sprites,
	//	then we're obviously looping...
	// 
	if (dwCount > kSpriteCount)
		{
		Display(true,
				"Chunk #%ld: circular link in chain type 0x%02lX.\n",
				m_nIndex, uint32(st));
		return false;
		}
	}

ASSERT(wSprite == 0xFFFF);
//
//	Check the count of sprites actually found vs. the count of sprites
//	expected.
//
if (dwCount != uint32(wExpected))
	{
	Display(true,
			"WARNING:  Chunk #%ld has a bad count for chain type 0x%02lX.\n",
			m_nIndex, uint32(st));
	}
else
	{
	Display(false, 
			"Chunk #%ld: %s sprites in chain type 0x%02lX.\n",
			m_nIndex, StrNum(dwCount), uint32(st));
	}

return true;
}	/* end of CSV6Data::verifyChain() */
/****************************************************************************
**
**	End of SV6Data.cpp
**
****************************************************************************/
