/****************************************************************************
**
**	File Name   : SV6Flags.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Tue 27 May '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the CSV6Flags class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Flags.cpp 4     5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Thu 06 Mar '03		1.00	Initial design & coding
**	Mon 10 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global (external) Constants/Variables
	----------------------------------------------------------------------	*/

/****************************************************************************
**
**	CSV6Flags::CSV6Flags() -- [CTOR]
**
*/
CSV6Flags::CSV6Flags() :	m_pFlags(NULL)
{
ASSERT( sizeof(SV6Flags) == 16 );
}	/* end of CSV6Flags::CSV6Flags() */
/****************************************************************************
**
**	CSV6Flags::~CSV6Flags() -- [DTOR]
**
*/
CSV6Flags::~CSV6Flags()
{
m_pFlags = NULL;	// don't free it, because it really points to m_pRawData
}	/* end of CSV6Flags::~CSV6Flags() */
/****************************************************************************
**
**	CSV6Flags::Apply() -- apply changes to this chunk
**
**	NOTES:
**		1.	This method must be called BEFORE CSV6Data::Apply() so that if the
**			game date is set, rides and sprites are updated accordingly.
*/
bool							// true if changes made, false otherwise
CSV6Flags::Apply(	
	Changes const &		ch)		// contains all changes to be applied
{
if (NULL_PTR(m_pFlags))
	return false;
//
//	Make a copy of the flags data
//
SV6Flags	flagsSave = *m_pFlags;
//
//	Change the date
//
if ((ch.SetDate.nDay || ch.SetDate.nMonth || ch.SetDate.nYear) &&
	setDate(ch.SetDate.nDay, ch.SetDate.nMonth, ch.SetDate.nYear))
	{
	Display(true,
			"Date changed to %s\n",
			StrDate(GetDay(), GetMonth(), GetYear()));
	}
//
//	Compare the saved copy to the current version...if there are any changes
//	then the flags have been changed and are "dirty".
//
return MEM_CMP(m_pFlags, &flagsSave, sizeof(SV6Flags)) ? true : false;
}	/* end of CSV6Flags::Apply() */
/****************************************************************************
**
**	CSV6Flags::Dump() -- dumps all flags information
**
*/
void
CSV6Flags::Dump(
	Options const &	opt) const
{
Display(opt.bDumpInfo,
		"Current date      : %15s\n",
		StrDate(GetDay(), GetMonth(), GetYear()));
}	/* end of CSV6Flags::Dump() */
/****************************************************************************
**
**	CSV6Flags::IsValidDate() -- validates a date
**
*/
bool							// true if valid, false otherwise
CSV6Flags::IsValidDate(
	int32		nDay,			// day (1..31)
	int32		nMonth,			// month (3..10)
	int32		nYear)			// year (1..kMaxYear)
{
const int32	kMonthLen[] = { 31, 30, 31, 30, 31, 31, 30, 31 };

return ((nYear >= kMinYear && nYear <= kMaxYear) &&
		(nMonth >= kMinMonth && nMonth <= kMaxMonth) &&
		(nDay >= 1 && nDay <= kMonthLen[nMonth - 3]));
}	/* end of CSV6Flags::IsValidDate() */
/****************************************************************************
**
**	CSV6Flags::Read() -- reads in a chunk of RCT data
**
*/
bool							// true on success, false otherwise
CSV6Flags::Read(
	int32		nIndex,			// chunk #
	DataPtr		pData,			// ptr to start of chunk
	uint32		dwDataLen,		// data bytes available
	uint32 &	dwRead)			// [out] size of chunk
{
m_nIndex = nIndex;		ASSERT(nIndex == 3);
	
if (!readChunk(pData, dwDataLen, dwRead))
	return false;

if (m_dwDataLen != sizeof(SV6Flags))
	{
	Display(true,
			"Chunk #%ld: invalid flags length (%s bytes).\n",
			m_nIndex, StrNum(m_dwDataLen));
	return false;
	}

//
//	Now validate the flags information
//
m_pFlags = reinterpret_cast<SV6FlagsPtr>( m_pRawData );

if (NULL_PTR(m_pFlags))
	{
	Display(true,
			"Chunk #%d: invalid flags.\n",
			m_nIndex);
	return false;
	}

return true;
}	/* end of CSV6Flags::Read() */
/****************************************************************************
**
**	CSV6Flags::setDate() -- set the current day/month/year
**
*/
bool							// true if set, false otherwise
CSV6Flags::setDate(
	int32		nDay,				// day (1..31)
	int32		nMonth,				// month (3..10)
	int32		nYear)				// year (1..kMaxYear)
{
if (GOOD_PTR(m_pFlags) &&
	IsValidDate(nDay, nMonth, nYear))
	{
	m_pFlags->wDay		= uint16(uint32(nDay * 0x8421) >> 4); 
	m_pFlags->wAbsMonth	= uint16((nMonth - 3) + ((nYear - 1) << 3));

	return true;
	}

return false;
}	/* end of CSV6Flags::setDate() */
/****************************************************************************
**
**	End of SV6Flags.cpp
**
****************************************************************************/
