/****************************************************************************
**
**	File Name   : SV6Sprite.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Tue 27 May '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the CSV6Sprite class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Sprite.cpp 6     5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Thu 06 Mar '03		1.00	Initial design & coding
**	Tue 11 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global (external) Constants/Variables
	----------------------------------------------------------------------	*/

/*	----------------------------------------------------------------------
	Local (static) Constants/Variables
	----------------------------------------------------------------------	*/
static const uint8		kMaxEnergy		= 0x80;
static const uint8		kMaxHappiness	= 0xFF;
static const uint8		kMaxNausea		= 0xFF;
static const uint8		kMaxTolerance	= 0x03;
/*	----------------------------------------------------------------------
	Local (static) Functions
	----------------------------------------------------------------------	*/
static TextPtr			getImageStr(int32);
static TextPtr			getTypeStr(int32);
/****************************************************************************
**
**	CSV6Sprite::CSV6Sprite() -- [CTOR]
**
*/
CSV6Sprite::CSV6Sprite() :	m_pSprite(NULL)
{
ASSERT( sizeof(SV6Sprite) == 0x100 );		// 256 bytes
}	/* end of CSV6Sprite::CSV6Sprite() */
/****************************************************************************
**
**	CSV6Sprite::~CSV6Sprite() -- [DTOR]
**
*/
CSV6Sprite::~CSV6Sprite()
{
m_pSprite = NULL;	// don't free it, because it really points to m_pRawData
}	/* end of CSV6Sprite::~CSV6Sprite() */
/****************************************************************************
**
**	CSV6Sprite::Apply() -- apply changes to the sprite information
**
*/
bool							// true if changes made, false otherwise
CSV6Sprite::Apply(
	Changes const &	ch,			// changes to apply
	uint32)						// [ignored] sprite #
{
if (NULL_PTR(m_pSprite))
	return false;
//
//	Make a copy of this sprite so that we can tell later if it was modified.
//
SV6Sprite	spriteSave = *m_pSprite;
//
//	buff up peep -- likes high intensity rides + high nausea tolerance
//
if (ch.bPeepsBuff && IsPeep())
	{
	m_pSprite->xNausea				= 0x00;				// not sick
	m_pSprite->xNauseaTolerance		= kMaxTolerance;	// high nausea tolerance
	m_pSprite->xPreferredIntensity	= 0xF0;				// likes all the rides
	}
//
//	give peep cash -- stored as # of dimes so multiply dollars * 10
//
if (ch.dwPeepCash && IsPeep())
	m_pSprite->dwCash = (ch.dwPeepCash - 1) * 10;
//
//	make peep happy -- full of energy, not sick, don't have to pee, etc.
//
if (ch.bPeepsHappy && IsPeep())
	{
	m_pSprite->xEnergy		= kMaxEnergy;			// full of energy
	m_pSprite->xHappiness	= kMaxHappiness;		// happy
	m_pSprite->xNausea		= 0x00;					// not sick
	m_pSprite->xBathroom	= 0x00;					// don't have to pee
	m_pSprite->xVandal		= 0x00;					// not a vandal anymore
	}
//
//	rob peep -- take all a peep's purchases away
//
if (ch.bPeepsRobbed && IsPeep())
	{
	m_pSprite->dwPurchases	= 0;
	m_pSprite->xGraphic		= 0x00;	// use "not carrying anything" image
	}
//
//	starve peep -- make peep hungry and thirsty
//
if (ch.bPeepsStarve && IsPeep())
	{
	m_pSprite->xHunger	= 0x00;					// hungry (0% full)
	m_pSprite->xThirst	= 0x00;					// thirsty
	}
//
//	Compare the saved copy to the current version...if there are any changes
//	then the sprite has been changed and is "dirty".
//
return MEM_CMP(m_pSprite, &spriteSave, sizeof(SV6Sprite)) ? true : false;
}	/* end of CSV6Sprite::Apply() */
/****************************************************************************
**
**	CSV6Sprite::Dump() -- display all the sprite information
**
*/
void
CSV6Sprite::Dump(
	Options const &	opt,
	uint32			dwSprite) const
{
if (IsEmpty())
	return;

ASSERT_PTR(m_pSprite, sizeof(SV6Sprite));

DisplayBar(opt.bDumpSprites, '-', opt.nOutputWidth);

if (IsPeep())
	{
	Display(opt.bDumpSprites,
			"0x%04lX Peep #%4lu %-25s Cash: %8s",
			dwSprite, m_pSprite->dwGuestNum, 
			getImageStr(m_pSprite->xGraphic),
			StrCash(m_pSprite->dwCash, 10));

	Display(opt.bDumpSprites,
			" E:%3s%% H:%3s%% N:%3s%%",
			StrNum(uint32(m_pSprite->xEnergy) * 100, kMaxEnergy),
			StrNum(uint32(m_pSprite->xHappiness) * 100, kMaxHappiness),
			StrNum(uint32(m_pSprite->xNausea) * 100, kMaxNausea));
	}
else if (IsStaff())
	{
	Display(opt.bDumpSprites,
			"0x%04lX %-36s",
			dwSprite, getImageStr(m_pSprite->xGraphic));
	}
else
	{
#if	0
	if (GetType() == ST_VEHICLE)
		DisplayHex(opt.bDumpSprites, DataPtr(m_pSprite), sizeof(SV6Sprite));
	else
#endif
	Display(opt.bDumpSprites,
			"0x%04lX %-16s ",
			dwSprite, getTypeStr(m_pSprite->xType));
	}

Display(opt.bDumpSprites, kStrNewLn);
}	/* end of CSV6Sprite::Dump() */
/****************************************************************************
**
**	getImageStr() -- [STATIC] returns a string describing the sprite image.
**
*/
static TextPtr					// ptr to text string
getImageStr(
	int32	nGraphics)			// graphics image (0x00..0xFF)
{
switch (nGraphics)
	{
	case 0x00:	return "Guest";

	case 0x01:	return "Handyman";
	case 0x02:	return "Mechanic";
	case 0x03:	return "Security";

	case 0x04:	return "Entertainer (Panda)";
	case 0x05:	return "Entertainer (Tiger)";
	case 0x06:	return "Entertainer (Elephant)";
	case 0x07:	return "Entertainer (Roman)";
	case 0x08:	return "Entertainer (Gorilla)";
	case 0x09:	return "Entertainer (Snowman)";
	case 0x0A:	return "Entertainer (Knight)";
	case 0x0B:	return "Entertainer (Astronaut)";
//
//	These values are from RCT1, but are not all correct for RCT2.
//
#if	0
	case 0x10:	return "Guest w/balloon";
	case 0x11:	return "Guest w/cotton candy";
	case 0x12:	return "Guest w/umbrella";
	case 0x13:	return "Guest w/pizza";

	case 0x15:	return "Guest w/popcorn";
	case 0x16:	return "Guest w/arms crossed";
	case 0x17:	return "Guest w/head down";
	case 0x18:	return "Guest w/hand over mouth";
	case 0x19:	return "Guest w/green face";
	case 0x1A:	return "Guest w/hand over crotch";
	case 0x1B:	return "Guest w/hat";
	case 0x1C:	return "Guest w/burger";
	case 0x1D:	return "Guest w/tentacle";
	case 0x1E:	return "Guest w/candy apple";
#endif

	default:
		break;
	}
/*
**	Format a generic string and return it
*/
static Text	szGraphic[kStrTmpLen+1];

MEM_ZERO(szGraphic, sizeof(szGraphic));
(void) STR_SPRINTF(szGraphic, kStrTmpLen, "Guest (0x%02X)", nGraphics);

return szGraphic;
}	/* end of getTypeStr() */
/****************************************************************************
**
**	getTypeStr() -- [STATIC] returns a string describing the sprite type.
**
*/
static TextPtr					// ptr to text string
getTypeStr(
	int32	nType)				// type (0x00..0xFF)
{
switch (nType)
	{
	case 0x00:	return "Vehicle";
	case 0x01:	return "Guest/Staff";
	case 0x02:	return "Duck/Flying Text";
	case 0x03:	return "Vomit/Trash";

	default:
		break;
	}
/*
**	Format a generic string and return it
*/
static Text	szType[kStrTmpLen+1];

MEM_ZERO(szType, sizeof(szType));
(void) STR_SPRINTF(szType, kStrTmpLen, "Type 0x%02X", nType);

return szType;
}	/* end of getTypeStr() */
/****************************************************************************
**
**	End of SV6Sprite.cpp
**
****************************************************************************/
