/****************************************************************************
**
**	File Name   : SV6Map.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Thu 13 Mar '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the CSV6Map class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Map.cpp 4     3/13/03 12:57p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Tue 04 Mar '03		1.00	Initial design & coding
**	Thu 13 Mar '03		1.01	Made code more portable (less Win32 stuff)
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global (external) Constants/Variables
	----------------------------------------------------------------------	*/

/*	----------------------------------------------------------------------
	Local (static) Constants/Definitions
	----------------------------------------------------------------------	*/
#define	TILE_TYPE(pm)	(((pm)->xData[0] >> 2) & 0x0F)

#define	IS_TRASHCAN(n)	((n) == 0x03)

static const TextPtr	kPathItems[] =	{
						/* 0x00 */		kStrEmpty,
						/* 0x01 */		"Lamp",
						/* 0x02 */		"Triple Lamp",
						/* 0x03 */		"Trash Can",
						/* 0x04 */		"Bench",
#if	0	// these differ from one *.SV6 file to the next?
						/* 0x05 */		"Jumping Fountain",
						/* 0x06 */		"Mine Trash Can",
						/* 0x07 */		"Egyptian Lamp",
						/* 0x08 */		"Jungle Bench",
						/* 0x09 */		"Egyptian Bench",
						/* 0x0A */		"Queue TV",
						/* 0x0B */		"Daisy Lamp",
						/* 0x0C */		"Pirate Lamp",
						/* 0x0D */		"Stone Trash Can",
						/* 0x0E */		"Metal Trash Can",
						/* 0x0F */		kStrEmpty,
						/* 0x10 */		"Jumping Snowballs"
#endif
										};

#define	ITEM_STR(n)		(((n) < ARRAY_DIM(kPathItems)) ? kPathItems[n] : kStrEmpty)

/****************************************************************************
**
**	CSV6Map::CSV6Map() -- [CTOR]
**
*/
CSV6Map::CSV6Map() :	m_pMap(NULL),
						m_dwMapEntries(0)
{
// STUB FUNCTION
}	/* end of CSV6Map::CSV6Map() */
/****************************************************************************
**
**	CSV6Map::~CSV6Map() -- [DTOR]
**
*/
CSV6Map::~CSV6Map()
{
m_pMap = NULL;		// don't delete this...it actually points to m_pRawData
}	/* end of CSV6Map::~CSV6Map() */
/****************************************************************************
**
**	CSV6Map::Apply() -- apply changes to this chunk
**
*/
bool							// true if changes made, false otherwise
CSV6Map::Apply(	
	Changes const &		ch)		// contains all changes to be applied
{
PANIC_IF_NULL(m_pMap);
//
//	Mow the grass
//
if (ch.bMakeTidy)
	{
	SV6MapEntryPtr	pMap;
	uint32			dwEmpty,
					dwFixed,
					dwMowed;

	dwEmpty = dwFixed = dwMowed = 0;
	for (uint32 dwx = 0; dwx < m_dwMapEntries; ++dwx)
		{
		pMap = &m_pMap[dwx];
		if (::IsBadWritePtr(pMap, sizeof(SV6MapEntry)))
			throw uint32( ERROR_INVALID_DATA );

		switch (TILE_TYPE(pMap))
			{
			//
			//	For surface tiles, the high nybble of byte 7 is the ownership, 
			//	the high 3 bits of byte 5 indicates the terrain type, and the 
			//	low 3 bits of byte 6 store the grass height.
			//
			case 0x00:		// surface tiles
				if ((pMap->xData[5] & 0xE0) == 0x00	&&	// is it grass terrain?
					(pMap->xData[6] & 0x07) != 0x00 &&	// does it need mowing?
					(pMap->xData[7] & 0x20))			// do we own the tile?
					{
					dwMowed++;
					pMap->xData[6] &= 0xF8;	// ~0x07 or 1111 1000
					}
				break;
			//
			//	For path tiles, we look at byte 5 to determine if there 
			//	are any decorations, i.e., benches, lamps, trash cans, etc.
			//
			case 0x01:		// path/road tiles
				{
				uint8	xItem = pMap->xData[5];
				//
				//	Check for full trash cans: the bottom 2 bits of byte 7 
				//	are 0x00 ("overflowing") through 0x03 ("empty").
				//
				if (IS_TRASHCAN(xItem) && (pMap->xData[7] & 0x03) < 0x03)
					{
					dwEmpty++;
					pMap->xData[7] |= 0x03;		// empty the trash
					}
				//
				//	Based on tests, the "vandalized" state is stored in bit 2
				//	of the high nybble of byte 1 (0 = normal, 1 = vandalized)
				//
				if ((xItem > 0x00 && xItem < 0x80) &&
					(pMap->xData[1] & 0x20))	// true if vandalized?
					{
					dwFixed++;
					pMap->xData[1] &= 0xDF;	// ~0x20
					}
				}
				break;

			default:		
				break;
			}
		}

	if (dwEmpty > 0)
		{
		m_bDirty = true;
		Display(true,
				"Emptied %s trash cans.\n",
				StrNum(dwEmpty));
		}

	if (dwFixed > 0)
		{
		m_bDirty = true;
		Display(true,
				"Fixed %s vandalized items.\n",
				StrNum(dwFixed));
		}

	if (dwMowed > 0)
		{
		m_bDirty = true;
		Display(true,
				"Mowed %s tiles.\n",
				StrNum(dwMowed));
		}
	}

return m_bDirty;
}	/* end of CSV6Map::Apply() */
/****************************************************************************
**
**	CSV6Map::Dump() -- dumps all map information
**
*/
void
CSV6Map::Dump(
	Options const &	opt) const
{
if (m_dwMapEntries == 0)
	{
	Display(opt.bDumpMap, "No map tiles!\n");
	return;
	}

PANIC_IF_NULL(m_pMap);
//
//	Now walk the map entries
//
SV6MapEntryPtr	pMap;

DisplayBar(opt.bDumpMap, '=', opt.nOutputWidth);
Display(opt.bDumpMap,
		"Map Tile  Item  Description       Busted\n"
		"========================================\n");

for (uint32	dwx = 0; dwx < m_dwMapEntries; ++dwx)
	{
	pMap = &m_pMap[dwx];

	switch (TILE_TYPE(pMap))
		{
		case 0x00:		// surface tiles
			break;

		case 0x01:		// path/road tiles
			//
			//	For path tiles, we look at byte 5 to determine if there 
			//	are any decorations, i.e., benches, lamps, trash cans, etc.
			//
			if (pMap->xData[5] != 0x00 && pMap->xData[5] != 0x80)
				{
				Display(opt.bDumpMap,
						"%8s  0x%02X  %-20s %c\n",  
						StrNum(dwx), 
						pMap->xData[5], 
						ITEM_STR(pMap->xData[5]),
						((pMap->xData[1] & 0x20) ? '*' : ' '));

//				DisplayHex(opt.bDumpMap, pMap->xData, sizeof(SV6MapEntry));
				}
			break;

		default:		
			break;
		}
	}

DisplayBar(opt.bDumpMap, '=', opt.nOutputWidth);
Display(opt.bDumpMap, 
		"%s map entries occupying %s bytes\n\n",
		StrNum(m_dwMapEntries), StrNum(m_dwDataLen));
}	/* end of CSV6Map::Dump() */
/****************************************************************************
**
**	CSV6Map::Read() -- reads in a chunk of RCT data
**
*/
bool							// true if read successfully, false otherwise
CSV6Map::Read(
	int32		nIndex,			// chunk #
	DataPtr		pData,			// ptr to start of chunk
	uint32		dwDataLen,		// data bytes available
	uint32 &	dwRead)			// [out] size of chunk
{
m_nIndex = nIndex;		ASSERT(nIndex == 4);
	
if (!readChunk(pData, dwDataLen, dwRead))
	return false;

m_dwMapEntries = m_dwDataLen / sizeof(SV6MapEntry);
//
//	Make sure we get at least an integral number of map entries
//
if ((m_dwMapEntries * sizeof(SV6MapEntry)) != m_dwDataLen)
	{
	Display(true,
			"Chunk #%ld: invalid data length (%s bytes).\n",
			m_nIndex, StrNum(m_dwDataLen));
	return false;
	}
else
	{
	Display(false,
			"Chunk #%ld contains %s map entries (%s bytes).\n",
			m_nIndex, StrNum(m_dwMapEntries), StrNum(m_dwDataLen));
	}

m_pMap = reinterpret_cast<SV6MapEntryPtr>( m_pRawData );

return true;
}	/* end of CSV6Map::Read() */
/****************************************************************************
**
**	End of SV6Map.cpp
**
****************************************************************************/
