/****************************************************************************
**
**	File Name   : startup.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Wed 28 May '03
**	          by: ,,,^..^,,,
**
**	Description : Parses the command line arguments and populates theChanges
**					and theOptions with the appropriate values.
**
**	Change Log	: $Header: /rctsgm/src/startup.cpp 10    5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Fri 07 Mar '03		1.00	Initial design & coding
**	Wed 12 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**	Wed 28 May '03		1.1		Added startup dialog
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global Variables -- declared in main.cpp
	----------------------------------------------------------------------	*/
extern Changes			theChanges;
extern Options			theOptions;
/*	----------------------------------------------------------------------
	Local Macros
	----------------------------------------------------------------------	*/
static const TextPtr	kLogFileName	= "rctsgm.log";

#define	IS_FLAG(p)		(*(p) == '-' || *(p) == '/')
/*	----------------------------------------------------------------------
	Local Function Prototypes
	----------------------------------------------------------------------	*/
static void				displayCopyright(void);
static void				displayHelp(void);
static bool				parseFlag(StringPtr);

inline static int32	
strToLong(TextPtr psz)
{
return STR_EMPTY(psz) ? 0 :
		strtol(psz, NULL, (StrCmpPartial(psz, "0x") == CSTR_EQUAL) ? 16 : 10);
}	/* end of strToLong() */
/****************************************************************************
**
**	ParseCmdLine() -- parses the command line.
**
*/
uint32							// NO_ERROR on success, error code otherwise
ParseCmdLine(
	int32			nArgs,		// # of command line arguments
	const StringPtr	pszArgs[])	// ptr to command line arguments
{
MEM_ZERO(&theChanges, sizeof(Changes));
MEM_ZERO(&theOptions, sizeof(Options));
//
//	Initialize our global setings
//
theOptions.nOutputWidth = 79;
theOptions.pLogFile = fopen(kLogFileName, "w+t");
if (NULL_PTR(theOptions.pLogFile))
	{
	Display(true,
			"Could not open log file \"%s\" for output.\n",
			kLogFileName);
	}
/*
**	Parse command line arguments
*/
if (nArgs == 1)	// no command line arguments?
	{
	theOptions.bWaitAtExit = true;
	return DoStartupDialog();
	}

for (int32 nArg = 1; nArg < nArgs; ++nArg)
	{
	StringPtr	pszArg = pszArgs[nArg];

	if (STR_EMPTY(pszArg))
		continue;	// skip NULL arguments (shouldn't ever see any)

	if (IS_FLAG(pszArg))
		{
		//
		// Skip over multiple flag markers so that we don't barf on GNU-style
		// flags, i.e., --help
		//
		while (IS_FLAG(pszArg))
			pszArg++;

		if (!parseFlag(pszArg))
			{
			Display(true,
					"Unknown flag ignored: \"%s\".\n", 
					pszArg);
			}
		}
	else if (STR_LEN(theOptions.szMask) < 1)
		{
		(void) STR_COPY(theOptions.szMask, pszArg, MAX_PATH);
		}
	else
		{
		Display(true,
				"Extra parameter ignored: \"%s\".\n", 
				pszArg);
		}
	}
/*
**	Now vet everything to make sure the user doesn't do something stupid.
*/
if (STR_LEN(theOptions.szMask) < 1)	// no file specified?
	STR_COPY(theOptions.szMask, "*.SV6", MAX_PATH);

return NO_ERROR;
}	/* end of ParseCmdLine() */
/****************************************************************************
**
**	displayCopyright() --	
**
*/
static void
displayCopyright(void)
{
#include "copyright.h"

DisplayBar(true, '*', theOptions.nOutputWidth);
Display(true, szCopyright);
DisplayBar(true, '*', theOptions.nOutputWidth);
}	/* end of displayCopyright() */
/****************************************************************************
**
**	displayHelp() --	
**
*/
static void
displayHelp(void)
{
Display(true,
		"RCTSGM -- RCT2 Save Game Modifier v%s\n"
		"Brought to you by Humble Programmer (humble_programmer@yahoo.com)    ,,,^..^,,,\n"
		"(internal data structures generously provided by \"ja227\" of www.rct2.com)\n\n"
		"Usage: rctsgm [flags] file.SV6\n"
		"Flags\n"
		"  /?   --help                Display this usage summary\n"
		"  /c   --copyright           Display the copyright information\n"
		"\n",
		kStrVersion);

Display(true,
		"Dump Commands:\n"
		"  /d   --dump                Dump all information\n"
		"  /di  --dumpinfo            Dump game information\n"
		"  /dm  --dumpmap             Dump all map information\n"
		"  /do  --dumpobjects         Dump custom object information\n"
		"  /dr  --dumprides           Dump ride information\n"
		"  /ds  --dumpsprites         Dump sprite information\n"
		"\n");

Display(true,
		"Make Commands:\n"
		"  /mf  --makefree            Makes everything free (no cash required)\n"
		"  /mm  --makemoney           Make park and ride fees modifiable\n"
		"  /mt  --maketidy            Remove all trash/vomit and mow grass\n"
		"\n");

Display(true,
		"Peep Commands:\n"
		"  /pb  --peepsbuff           Make all peeps \"buff\"\n"
		"  /pc  --peepscash=<value>   Give each peep <value> in cash\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n"
		"  /pg  --peepsgone           Make all peeps disappear\n"
		"  /ph  --peepshappy          Make all peeps happy\n"
		"  /pr  --peepsrobbed         Take all peeps purchases away\n"
		"  /ps  --peepsstarved        Make all peeps hungry & thirsty\n"
		"\n",
		StrCash(kMinPeepCash), StrCash(kMaxPeepCash), StrCash(kMaxPeepCash));

Display(true,
		"Rides commands:\n"
		"  /rf  --ridesfixed          Make all rides brand new\n"
		"  /rn  --ridesnormal         Make all rides normal\n"
		"  /rq  --ridesquiet          Make all rides stop playing music\n"
		"\n");

Display(true,
		"Set Commands:\n"
		"  /sc  --setcash=<value>     Set cash amount to <value>\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n",
		StrCash(kMinCash), StrCash(kMaxCash), StrCash(kMaxCash));

Display(true,
		"  /sd  --setdate=<dd-mm-yy>  Set the date to <dd-mm-yy>\n"
		"                             <dd> must be between 1 and 31\n"
		"                             <mm> must be between 3 and 10\n"
		"                             <yy> must be between 1 and %d\n"
		"                             defaults to 1-Mar-01 if <dd-mm-yy> is omitted\n",
		kMaxYear);

Display(true,
		"  /si  --setintrate=<value>  Set interest rate to <value>\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n",
		StrNum(kMinIntRate), StrNum(kMaxIntRate), StrNum(kMinIntRate));

Display(true,
		"  /sl  --setloan=<value>     Set loan amount to <value>\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n",
		StrCash(kMinLoan), StrCash(kMaxLoan), StrCash(kMinLoan));

Display(true,
		"  /sm  --setmaxloan=<value>  Set maximum loan amount to <value>\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n",
		StrCash(kMinLoan), StrCash(kMaxLoan), StrCash(kMaxLoan));

Display(true,
		"  /sr  --setrating=<value>   Set park rating to <value>\n"
		"                             <value> must be between %s and %s\n"
		"                             defaults to %s if <value> is omitted\n"
		"\n",
		StrNum(kMinRating), StrNum(kMaxRating), StrNum(kMaxRating));

Display(true,
		"Other commands:\n"
		"  /e   --expand              Make an uncompressed copy of the input file\n"
		"  /f   --fix                 Tries to fix invalid data and/or objects\n"
		"  /q   --quiet               Quiet mode (no screen output)\n"
		"  /v   --verbose             Maximum verbosity\n"
		"  /w   --wait                Wait at exit\n"
		"\n",
		kStrVersion);
}	/* end of displayHelp() */
/****************************************************************************
**
**	parseFlag() --	
**
**	NOTES:
**		1.	For some reason, PC-Lint insists that pszArg could be made const,
**			but we later pass it to strrchr() and use the resulting ptr to 
**			modify the string, so the compiler barfs if it's constant.
*/
static bool						// true if flag recognized, false otherwise
parseFlag(
	StringPtr	pszArg)			//lint -esym(818,pszArg) See Note #1
{
int32		nValue;
StringPtr	pszValue;
bool		bOK = true;

if (STR_EMPTY(pszArg))
	return false;
//
//	Look for a value on the end of the flag, i.e., --size=16
//  Both the colon (:) and the equal sign (=) are allowed. 
//
pszValue = STR_RCHR(pszArg, ':');
if (NULL_PTR(pszValue))
	pszValue = STR_RCHR(pszArg, '=');

if (GOOD_PTR(pszValue))
	*pszValue++ = '\0';	// clobber original separator
/*
**	Generic flags
*/
if (*pszArg == '?' ||
	StrCmpPartial(pszArg, "help") == CSTR_EQUAL)
	{
	displayHelp();
	}
else if (StrCmpPartial(pszArg, "copyright") == CSTR_EQUAL)
	{
	displayCopyright();
	}
else if (StrCmpPartial(pszArg, "expand") == CSTR_EQUAL)
	{
	theOptions.bExpand = true;
	}
else if (StrCmpPartial(pszArg, "fix") == CSTR_EQUAL)
	{
	theChanges.bFix = true;
	}
else if (StrCmpPartial(pszArg, "quiet") == CSTR_EQUAL)
	{
	theOptions.bQuiet = true;
	}
else if (StrCmpPartial(pszArg, "verbose") == CSTR_EQUAL)
	{
	theOptions.bVerbose = true;
	Display(false, "Maximum verbosity.\n");
	}
else if (StrCmpPartial(pszArg, "wait") == CSTR_EQUAL)
	{
	theOptions.bWaitAtExit = true;
	}
/*
**	DUMP commands
*/
else if (StrCmpPartial(pszArg, "dump") == CSTR_EQUAL)
	{
	theOptions.bDumpInfo	= true;
	theOptions.bDumpMap		= true;
	theOptions.bDumpObjects	= true;
	theOptions.bDumpRides	= true;
	theOptions.bDumpSprites	= true;
	}
else if (StrCmpPartial(pszArg, "di") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "dumpinfo") == CSTR_EQUAL)
	{
	theOptions.bDumpInfo = true;
	}
else if (StrCmpPartial(pszArg, "dm") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "dumpmap") == CSTR_EQUAL)
	{
	theOptions.bDumpMap = true;
	}
else if (StrCmpPartial(pszArg, "do") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "dumpobjects") == CSTR_EQUAL)
	{
	theOptions.bDumpObjects = true;
	}
else if (StrCmpPartial(pszArg, "dr") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "dumprides") == CSTR_EQUAL)
	{
	theOptions.bDumpRides = true;
	}
else if (StrCmpPartial(pszArg, "ds") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "dumpsprites") == CSTR_EQUAL)
	{
	theOptions.bDumpSprites = true;
	}
/*
**	MAKE commands
*/
else if (StrCmpPartial(pszArg, "mf") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "makefree") == CSTR_EQUAL)
	{
	theChanges.bMakeFree = true;
	}
else if (StrCmpPartial(pszArg, "mm") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "makemoney") == CSTR_EQUAL)
	{
	theChanges.bMakeMoney = true;
	}
else if (StrCmpPartial(pszArg, "mt") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "maketidy") == CSTR_EQUAL)
	{
	theChanges.bMakeTidy = true;
	}
/*
**	PEEP commands
*/
else if (StrCmpPartial(pszArg, "pb") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepsbuff") == CSTR_EQUAL)
	{
	theChanges.bPeepsBuff = true;
	}
else if (StrCmpPartial(pszArg, "pc") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepscash") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMaxPeepCash : strToLong(pszValue);
	if (nValue >= kMinPeepCash && nValue <= kMaxPeepCash)
		theChanges.dwPeepCash = uint32(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --peepscash value (%s)\n",
				StrNum(nValue));
		}
	}
else if (StrCmpPartial(pszArg, "pg") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepsgone") == CSTR_EQUAL)
	{
	theChanges.bPeepsGone = true;
	}
else if (StrCmpPartial(pszArg, "ph") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepshappy") == CSTR_EQUAL)
	{
	theChanges.bPeepsHappy = true;
	}
else if (StrCmpPartial(pszArg, "pr") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepsrobbed") == CSTR_EQUAL)
	{
	theChanges.bPeepsRobbed = true;
	}
else if (StrCmpPartial(pszArg, "ps") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "peepsstarved") == CSTR_EQUAL)
	{
	theChanges.bPeepsStarve = true;
	}
/*
** RIDE Commands
*/
else if (StrCmpPartial(pszArg, "rf") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "ridesfixed") == CSTR_EQUAL)
	{
	theChanges.bRidesFixed = true;
	}
else if (StrCmpPartial(pszArg, "rp") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "ridesplain") == CSTR_EQUAL)
	{
	theChanges.bRidesPlain = true;
	}
else if (StrCmpPartial(pszArg, "rq") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "ridesquiet") == CSTR_EQUAL)
	{
	theChanges.bRidesQuiet = true;
	}
/*
**	SET commands
*/
else if (StrCmpPartial(pszArg, "sc") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setcash") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMaxCash : strToLong(pszValue);
	if (nValue >= kMinCash && nValue <= kMaxCash)
		theChanges.dwCash = uint32(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --setcash value (%s)\n",
				StrNum(nValue));
		}
	}
else if (StrCmpPartial(pszArg, "sd") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setdate") == CSTR_EQUAL)
	{
	if (NULL_PTR(pszValue))
		{
		theChanges.SetDate.nDay		= 1;
		theChanges.SetDate.nMonth	= 3;
		theChanges.SetDate.nYear	= 1;
		}
	else 
		{
		int32		nDay, nMonth, nYear;

		if (sscanf(pszValue, "%ld-%ld-%ld", &nDay, &nMonth, &nYear) == 3 &&
			CSV6Flags::IsValidDate(nDay, nMonth, nYear))
			{
			theChanges.SetDate.nDay		= nDay;
			theChanges.SetDate.nMonth	= nMonth;
			theChanges.SetDate.nYear	= nYear;
			}
		else
			{
			Display(true,
					"Ignoring invalid --setdate value \"%s\"\n",
					pszValue);
			}
		}
	}
else if (StrCmpPartial(pszArg, "si") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setintrate") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMinIntRate : strToLong(pszValue);
	if (nValue >= kMinIntRate && nValue <= kMaxIntRate)
		theChanges.wIntRate = uint16(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --setintrate value (%s)\n",
				StrNum(nValue));
		}
	}
else if (StrCmpPartial(pszArg, "sl") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setloan") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMinLoan : strToLong(pszValue);
	if (nValue >= kMinLoan && nValue <= kMaxLoan)
		theChanges.dwLoan = uint32(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --setloan value (%s)\n",
				StrNum(nValue));
		}
	}
else if (StrCmpPartial(pszArg, "sm") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setmaxloan") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMaxLoan : strToLong(pszValue);
	if (nValue >= kMinLoan && nValue <= kMaxLoan)
		theChanges.dwMaxLoan = uint32(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --setmaxloan value (%s)\n",
				StrNum(nValue));
		}
	}
else if (StrCmpPartial(pszArg, "sr") == CSTR_EQUAL ||
		 StrCmpPartial(pszArg, "setrating") == CSTR_EQUAL)
	{
	nValue = NULL_PTR(pszValue) ? kMaxRating : strToLong(pszValue);
	if (nValue >= kMinRating && nValue <= kMaxRating)
		theChanges.wParkRating = uint16(nValue) + 1;
	else
		{
		Display(true,
				"Ignoring invalid --setrating value (%s)\n",
				StrNum(nValue));
		}
	}
/*
**	Odds & Sods
*/
else if (StrCmpPartial(pszArg, "humble") == CSTR_EQUAL)
	{
	//
	//	These are few of my favorite things! 
	//
	theOptions.bQuiet		= true;
	theOptions.bVerbose		= true;

	theChanges.bMakeMoney	= true;
	theChanges.bMakeTidy	= true;

	theChanges.bPeepsBuff	= true;
	theChanges.bPeepsHappy	= true;
	theChanges.bPeepsRobbed	= true;
	theChanges.bPeepsStarve	= true;

	theChanges.bRidesFixed	= true;

	theChanges.dwCash		= kMaxCash + 1;
	theChanges.dwLoan		= kMinLoan + 1;
	theChanges.dwMaxLoan	= kMaxLoan + 1;
	theChanges.dwPeepCash	= kMaxPeepCash + 1;

	theChanges.SetDate.nDay		= 1;
	theChanges.SetDate.nMonth	= 3;
	theChanges.SetDate.nYear	= 1;
	}
else	//	unrecognized flag?
	bOK = false;

return bOK;
}	/* end of parseFlag() */
/****************************************************************************
**
**	End of startup.cpp
**
****************************************************************************/
