/****************************************************************************
**
**	File Name   : SV6Ride.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Thu 29 May '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of the CSV6Ride class.
**
**	Change Log	: $Header: /rctsgm/src/SV6Ride.cpp 9     5/27/03 4:13p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Thu 06 Mar '03		1.00	Initial design & coding
**	Wed 12 Mar '03		1.01	Made code more portable (less Win32 stuff)
**	Tue 27 May '03		1.02	Added --fix flag and logic
**	Thu 29 May '03		1.10	Added startup dialog
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global (external) Constants/Variables
	----------------------------------------------------------------------	*/
extern RCTFile			theFile;
/*	----------------------------------------------------------------------
	Local (static) Constants/Variables
	----------------------------------------------------------------------	*/
static const TextPtr	kRideStatus[] =	{
						/* 0x00 */		"Closed",
						/* 0x01 */		"Open",
						/* 0x02 */		"Testing"
										};
static const TextPtr	kRideTypes[] =	{
						/* 0x00 */		"Spiral RC",
						/* 0x01 */		"Stand-up RC",
						/* 0x02 */		"Suspended Swinging RC",
						/* 0x03 */		"Inverted RC",	
						/* 0x04 */		"Junior RC",	
						/* 0x05 */		"Miniature Railway",	
						/* 0x06 */		"Monorail",
						/* 0x07 */		"Mini Suspended RC",
						/* 0x08 */		"Boat Hire",	
						/* 0x09 */		"Wooden Wild RC",	
						/* 0x0A */		"Single-Rail RC",
						/* 0x0B */		"Car Ride",
						/* 0x0C */		"Launched Freefall",
						/* 0x0D */		"Bobsled RC",
						/* 0x0E */		"Observation Tower",
						/* 0x0F */		"Looping RC",
						/* 0x10 */		"Dinghy Slide",
						/* 0x11 */		"Mine Train RC",
						/* 0x12 */		"Chairlift",
						/* 0x13 */		"Hypercoaster",	
						/* 0x14 */		"Maze",	
						/* 0x15 */		"Spiral Slide",	
						/* 0x16 */		"Go Karts",
						/* 0x17 */		"Log Flume",
						/* 0x18 */		"River Rapids",	
						/* 0x19 */		"Bumper Cars ",	
						/* 0x1A */		"Pirate Ship",
						/* 0x1B */		"Swinging Inverter Ship",
						/* 0x1C */		"Food Stall",
						/* 0x1D */		"(0x1D)",
						/* 0x1E */		"Drink Stall",
						/* 0x1F */		"(0x1F)",
						/* 0x20 */		"Shop",
						/* 0x21 */		"Merry-Go-Round",
						/* 0x22 */		"(0x22)",
						/* 0x23 */		"Information Kiosk",	
						/* 0x24 */		"Bathroom",	
						/* 0x25 */		"Ferris Wheel",	
						/* 0x26 */		"Motion Simulator",
						/* 0x27 */		"3D Cinema",
						/* 0x28 */		"Top Spin",	
						/* 0x29 */		"Space Rings",	
						/* 0x2A */		"Reverse Freefall RC",
						/* 0x2B */		"Elevator",
						/* 0x2C */		"Vertical Drop RC",
						/* 0x2D */		"ATM",
						/* 0x2E */		"Twist",
						/* 0x2F */		"Haunted House",
						/* 0x30 */		"First Aid",
						/* 0x31 */		"Circus Show",
						/* 0x32 */		"Ghost Train",
						/* 0x33 */		"Twister RC",	
						/* 0x34 */		"Wooden RC",	
						/* 0x35 */		"Side-Friction RC",	
						/* 0x36 */		"Wild Mouse RC",
						/* 0x37 */		"Multi-Dimension RC",
						/* 0x38 */		"(0x38)",	
						/* 0x39 */		"Flying RC",	
						/* 0x3A */		"(0x3A)",
						/* 0x3B */		"Virginia Reel RC",
						/* 0x3C */		"Splash Boats",
						/* 0x3D */		"Mini Helicopters",
						/* 0x3E */		"Lay-down RC",
						/* 0x3F */		"Suspended Monorail",
						/* 0x40 */		"(0x40)",
						/* 0x41 */		"Reverser RC",
						/* 0x42 */		"Heartline Twister RC",
						/* 0x43 */		"Mini Golf",	
						/* 0x44 */		"Giga RC",	
						/* 0x45 */		"Roto-Drop",	
						/* 0x46 */		"Flying Saucers",
						/* 0x47 */		"Crooked House",
						/* 0x48 */		"Monorail Cycles",	
						/* 0x49 */		"Inverted Shuttle RC",	
						/* 0x4A */		"Water RC",
						/* 0x4B */		"Air Powered Vertical RC",
						/* 0x4C */		"Inverted Hairpin RC",
						/* 0x4D */		"Magic Carpet",
						/* 0x4E */		"Submarine Ride",
						/* 0x4F */		"River Rafts",
						/* 0x50 */		"(0x50)",
						/* 0x51 */		"Enterprise",
						/* 0x52 */		"(0x52)",
						/* 0x53 */		"(0x53)",	
						/* 0x54 */		"(0x54)",	
						/* 0x55 */		"(0x55)",	
						/* 0x56 */		"Inverted Impulse RC",
						/* 0x57 */		"Mini RC",
						/* 0x58 */		"Mine Ride",	
						/* 0x59 */		"(0x59)",	
						/* 0x5A */		"LIM Launched RC"
										};

#define	PRICE_STR(n)	((n) ? StrCash((n), 10) : "free")
#define	STATUS_STR(n)	(((n) < ARRAY_DIM(kRideStatus)) ? kRideStatus[n] : kStrEmpty)
#define	TYPE_STR(n)		(((n) < ARRAY_DIM(kRideTypes)) ? kRideTypes[n] : kStrEmpty)
/*	----------------------------------------------------------------------
	Local (static) Functions
	----------------------------------------------------------------------	*/

/****************************************************************************
**
**	CSV6Ride::CSV6Ride() -- [CTOR]
**
*/
CSV6Ride::CSV6Ride() :	m_pRide(NULL)
{
ASSERT( sizeof(SV6Ride) == 0x260 );		// 608 bytes

MEM_ZERO(m_szName, sizeof(m_szName)); 
}	/* end of CSV6Ride::CSV6Ride() */
/****************************************************************************
**
**	CSV6Ride::~CSV6Ride() -- [DTOR]
**
*/
CSV6Ride::~CSV6Ride()
{
m_pRide = NULL;	// don't free it, because it really points to m_pRawData
}	/* end of CSV6Ride::~CSV6Ride() */
/****************************************************************************
**
**	CSV6Ride::Apply() -- apply changes to the ride information
**
*/
bool							// true if changes made, false otherwise
CSV6Ride::Apply(
	Changes const &	ch,			// changes to apply
	uint32)						// [ignored] ride #
{
if (IsEmpty())
	return false;
//
//	Make a copy of the ride data
//
PANIC_IF_NULL(m_pRide);

SV6Ride	rideSave = *m_pRide;
//
//	Make the ride (and shop merchandise) free.  TODO: what about shops with
//	multiple items, like Kiosks?
//	
if (ch.bMakeFree)
	m_pRide->wTicketPrice = 0x00;
//
//	Check the "built" date, and fix it if necessary.  Also check the date if
//	the user has tried to modify the game date.
//
uint16	wMonthNow = theFile.GetAbsMonth();

if ((ch.bFix || ch.SetDate.nDay || ch.SetDate.nMonth || ch.SetDate.nYear) &&
	m_pRide->wMonthBuilt > wMonthNow)
	{
	m_pRide->wMonthBuilt = wMonthNow;
	}

if (IsShopOrStall())	// rest of changes only apply to rides
	return MEM_CMP(m_pRide, &rideSave, sizeof(SV6Ride)) ? true : false;	//lint !e668
//
//	If we're banishing peeps, we need to close the ride and empty the queue
//	lines.  11-Mar-03: Closing the ride doesn't seem to help fix the "stalled
//	ride" bug.  My current guess is the vehicles need to be reset as well.
//
if (ch.bPeepsGone)
	{
	m_pRide->xRiderCount = 0x00;	// nobody on the ride any more
	for (uint32 dwx = 0; dwx < kRideStations; ++dwx)
		{
		m_pRide->xQueuedGuests[dwx] = 0x00;
		m_pRide->wLastGuest[dwx] = 0xFFFF;
		}
	}
//
//	repair ride -- make 100% reliable and brand new (built this month)
//
if (ch.bRidesFixed)
	{
	m_pRide->xMechanicStatus		= 0x00;	// no mechanic fixing the ride
	m_pRide->xBreakDown				= 0xFF;	// not broken down any more
	m_pRide->xReliabilityCounter	= 0xFF;	// max out the reliability counter
	m_pRide->xReliability			= 100;	// now 100% reliable
	m_pRide->xTimeSinceInspection	= 0x00;	// just inspected
	m_pRide->wMonthBuilt			= wMonthNow;

	(void) setStatusBit(7, false);	// not broken down anymore
	(void) setStatusBit(10, false);	// never crashed
	}
//
//	Make ride plain, i.e., remove "six flags" and "indestructible" flags.
//
if (ch.bRidesPlain)
	{
	(void) setStatusBit(14, false);
	(void) setStatusBit(15, false);
	(void) setStatusBit(19, false);
	}

if (ch.bRidesQuiet)
	{
	//
	//	Setting .xMusicTheme to 0x21 will raise an exception if the user tries
	//	to turn the music back on.
	//
	m_pRide->xMusicType = 0xFF;		// no music selection
	(void) setStatusBit(13, false);	// turn off music flag
	}
//
//	Compare the saved copy to the current version...if there are any changes
//	then the ride has been changed and is "dirty".
//
return MEM_CMP(m_pRide, &rideSave, sizeof(SV6Ride)) ? true : false;	//lint !e668
}	/* end of CSV6Ride::Apply() */
/****************************************************************************
**
**	CSV6Ride::Dump() -- display all the ride information
**
*/
void
CSV6Ride::Dump(
	Options const &	opt,
	uint32			dwRide) const
{
if (IsEmpty())
	return;

ASSERT_PTR(m_pRide, sizeof(SV6Ride));

DisplayBar(opt.bDumpRides, '-', opt.nOutputWidth);
Display(opt.bDumpRides,
		"%4lu %-32s %8s %7s",
		dwRide, 
		TYPE_STR( m_pRide->xType ), 
		STATUS_STR( m_pRide->xRideStatus ),
		StrDate(m_pRide->wMonthBuilt));

if (getStatusBit(7))
	Display(opt.bDumpRides, " BrokenDown");
if (getStatusBit(7))
	Display(opt.bDumpRides, " Crashed");
if (getStatusBit(14) || getStatusBit(15))
	Display(opt.bDumpRides, " Indestructible");
if (getStatusBit(19))
	Display(opt.bDumpRides, " SixFlags");

Display(opt.bDumpRides,
		"\n     %-32s %8s ",
		m_szName, PRICE_STR( m_pRide->wTicketPrice ));

if (IsShopOrStall())
	{
	Display(opt.bDumpRides, kStrNewLn);
	}
else
	{
	//
	//	Some rides have garbage EIN ratings, so don't display them if any of the
	//	ratings have the MSB set.  We can test all 3 at once with a little bit of
	//	bit bashing...
	//
	if ((m_pRide->wExcitement | m_pRide->wIntensity | m_pRide->wNausea) & 0x8000)
		Display(opt.bDumpRides, " E: --.- I: --.- N: --.- ");
	else
		{
		Display(opt.bDumpRides,
				" E:%5s I:%5s N:%5s ",
				StrNum(m_pRide->wExcitement, 100, 2),
				StrNum(m_pRide->wIntensity, 100, 2),
				StrNum(m_pRide->wNausea, 100, 2));
		}

	Display(opt.bDumpRides,
			"R:%3u%%\n",
			m_pRide->xReliability);

#if	0
	Display(opt.bDumpRides, "     ");
	for (uint32 dwBit = 0; dwBit < 32; ++dwBit)
		{
		Display(opt.bDumpRides, (getStatusBit(dwBit) ? "1" : "0"));
		if (((dwBit + 1) % 8) == 0)			// every 8th bit...
			Display(opt.bDumpRides, " ");	// put in a space for clarity
		}
	Display(opt.bDumpRides, kStrNewLn);
#endif
	}
}	/* end of CSV6Ride::Dump() */
/****************************************************************************
**
**	CSV6Ride::Set() -- populate the object from the raw data.
**
*/
bool							// true if successful, false otherwise
CSV6Ride::Set(
	SV6RidePtr		pRide)		// ptr to ride data
{
if (NULL_PTR(pRide) || ::IsBadReadPtr(pRide, sizeof(SV6Ride)))
	return false;

m_pRide = pRide;
//
//	Build a copy of the name (the name field in the structure is NOT zero-
//	terminated.
//
uint16		wIndex = m_pRide->wStringIndex;
TextPtr		pStrMap = TextPtr( theFile.GetStringMap() );

MEM_ZERO(m_szName, sizeof(m_szName));

if ((wIndex & 0x8000) &&	// true if MSB is set
	GOOD_PTR(pStrMap))		// and we have a valid string map
	{
	wIndex = (wIndex & 0x3FF) * 32;	// use only lower 10 bits
	(void) STR_COPY(m_szName, &pStrMap[wIndex], kRideNameLen);
	}
else
	{
	(void) STR_SPRINTF(m_szName, kRideNameLen, "String #%u", wIndex);
	}

return true;
}	/* end of CSV6Ride::Set() */
/****************************************************************************
**
**	CSV6Ride::getStatusBit() -- returns one of the 24 status flags in the 
**		SV6Ride->xStatusFlags[] array.
**
****************************************************************************/
bool									// value of bit (false if invalid)
CSV6Ride::getStatusBit(
	int32		nFlag) const			// bit # to return (0..23)
{
return (GOOD_PTR(m_pRide) &&
		(nFlag >= 0 && nFlag < 24) &&
		(m_pRide->xStatusFlags[nFlag / 8] & (1 << (nFlag & 0x07))));
}	/* end of CSV6Ride::getStatusBit() */
/****************************************************************************
**
**	CSV6Ride::setStatusBit() -- sets/clears one of the 24 status flags in 
**		the SV6Ride->xStatusFlags[] array.
**
****************************************************************************/
bool							// true if valid set, false otherwise
CSV6Ride::setStatusBit(
	int32	nFlag, 				// bit # to set/clear (0..23)
	bool	bOn)
{
if (GOOD_PTR(m_pRide) && (nFlag >= 0 && nFlag < 24))
	{
	if (bOn)
		m_pRide->xStatusFlags[nFlag / 8] |= (1 << (nFlag & 0x07));
	else
		m_pRide->xStatusFlags[nFlag / 8] &= ~(1 << (nFlag & 0x07));

	return true;
	}

return false;
}	/* end of CSV6Ride::setStatusBit() */
/****************************************************************************
**
**	End of SV6Ride.cpp
**
****************************************************************************/
