/****************************************************************************
**
**	File Name   : utility.cpp
**
**	Project     : RCTSGM
**
**	Last Updated: Mon 10 Mar '03
**	          by: ,,,^..^,,,
**
**	Description : Implementation of global utility functions.
**
**	Change Log	: $Header: /rctsgm/src/utility.cpp 4     3/20/03 2:34p Neusel $
**
**		Date		 Version	Reason
**		====		 =======	======
**	Fri 07 Mar '03		1.00	Initial design & coding
**	Mon 10 Mar '03		1.01	Made code more portable (less Win32 stuff)
**
****************************************************************************/
#include "rctsgm.h"
#include "rctfile.h"

#undef	THIS_FILE
const TextPtr			THIS_FILE = __FILE__;
/*	----------------------------------------------------------------------
	Global Constants/Variables
	----------------------------------------------------------------------	*/
extern Options			theOptions;
/*	----------------------------------------------------------------------
	Local (static) Constants/Variables
	----------------------------------------------------------------------	*/
static const uint32		kTmpStrCount	= 8;
static const uint32		kTmpStrLength	= 42;

static uint32			s_uBuffer = 0;
static Text			s_cBuffer[kTmpStrCount][kTmpStrLength+1];
/*	----------------------------------------------------------------------
	Local Function Prototypes
	----------------------------------------------------------------------	*/

/****************************************************************************
**
**	Display()
**
*/
void
Display(
	bool		bAlways,		// true to always display, false if verbose
	TextPtr		pstrFormat,		// ptr to format string
	...)						// [optional] printf()-style parameters
{
if (GOOD_PTR(pstrFormat) && (bAlways || theOptions.bVerbose))
	{
	va_list		vaList;

	if (!theOptions.bQuiet)
		{
		va_start(vaList, pstrFormat);
		(void) vfprintf(stdout, pstrFormat, vaList);
		va_end(vaList);
		}

	if (GOOD_PTR(theOptions.pLogFile))
		{
		va_start(vaList, pstrFormat);
		(void) vfprintf(theOptions.pLogFile, pstrFormat, vaList);
		va_end(vaList);
		}
	}
}	/* end of Display() */
/****************************************************************************
**
**	DisplayBar()
**
*/
void
DisplayBar(
	bool	bAlways,			// true to draw
	Text	ch,					// character to draw
	int32	nCount)				// # if characters to draw
{
if (nCount > 0 && (bAlways || theOptions.bVerbose))
	{
	Text	szBar[kMaxOutputWidth+1];

	MEM_ZERO(szBar, sizeof(szBar));
	MEM_FILL(szBar, min(nCount, kMaxOutputWidth), ch);	//lint !e732
	
	Display(bAlways, szBar);
	Display(bAlways, kStrNewLn);
	}
}	/* end of DisplayBar() */
/****************************************************************************
**
**	DisplayHex()
**
*/
void
DisplayHex(
	bool		bAlways,		// true to always display, false if verbose
	DataPtr		pData,			// ptr to data
	uint32		dwBytes)		// # of bytes
{
if (GOOD_PTR(pData) && dwBytes > 0)
	{
	uint32	dwx,
			dwRow,
			dwSoFar;

	ASSERT_PTR(pData, dwBytes);

	for (dwSoFar = 0; dwSoFar < dwBytes; dwSoFar += dwRow)
		{
		if ((dwRow = dwBytes - dwSoFar) > 16)
			dwRow = 16;

		Display(bAlways, "  0x%04X : ", dwSoFar);

		for (dwx = 0; dwx < 16; ++dwx)
			{
			if (dwx < dwRow)
				Display(bAlways, "%02X ", pData[dwSoFar + dwx]);
			else
				Display(bAlways, "   ");
			}

		Display(bAlways, "| ");

		for (dwx = 0; dwx < dwRow; ++dwx)
			{
			Text	ch = pData[dwSoFar + dwx];

			Display(bAlways, "%c", isprint(ch) ? ch : '.');
			}

		Display(bAlways, kStrNewLn);
		}
	}
}	/* end of DisplayHex() */
/****************************************************************************
**
**	GetWindowHandle() -- wrapper around GetConsoleWindow(), which is only 
**		supported under Win2K or better, so we have to do some ugly dynamic 
**		loading/linking to see if we can use it or not.
**
*/
HWND
GetWindowHandle(void)
{
HWND	hwnd = NULL;
HMODULE	hModule = ::LoadLibrary("kernel32.dll");

if (GOOD_HND(hModule))
	{
	typedef HWND (WINAPI *PFN_GetConsoleWindow)(void);
	
	PFN_GetConsoleWindow	pfn = (PFN_GetConsoleWindow)::GetProcAddress(hModule, "GetConsoleWindow");

	if (GOOD_PTR(pfn))
		hwnd = pfn();

	::FreeLibrary(hModule);
	}

return hwnd;
}	/* end of GetWindowHandle() */
/****************************************************************************
**
**	StrCmpPartial() --	
**
*/
int32							// CSTR_EQUAL, CSTR_LESS_THAN, CSTR_GREATER_THAN
StrCmpPartial(
	TextPtr		pstr1,			// left string
	TextPtr		pstr2)			// right string
{
if (NULL_PTR(pstr1))
	return GOOD_PTR(pstr2) ? CSTR_LESS_THAN : CSTR_EQUAL;

if (NULL_PTR(pstr2))
	return GOOD_PTR(pstr1) ? CSTR_GREATER_THAN : CSTR_EQUAL;
/*
**	Now actually compare the strings
*/
int32	nCmp,
		nCmpLen;
size_t	sLen1 = STR_LEN(pstr1),
		sLen2 = STR_LEN(pstr2);

nCmpLen = (int) min(sLen1, sLen2);
nCmp = ::CompareString(LOCALE_USER_DEFAULT, 
						NORM_IGNORECASE | NORM_IGNORESYMBOLS | NORM_IGNOREWIDTH, 
						pstr1, nCmpLen, pstr2, nCmpLen);

API_PANIC_IF(nCmp == 0);

return nCmp;
}	/* end of StrCmpPartial() */
/****************************************************************************
**
**	StrCash() --	formats a binary long to a string with commas (replaces
**					ltoa() and printf()/sprintf() functions)
**
*/
StringPtr						// ptr to output string (buffer)
StrCash(
	int32		nValue,			// value to convert
	int32		nDivisor)		// divisor
{
StringPtr	pszResult;
Text		szBuffer[kTmpStrLength+1];

pszResult = s_cBuffer[s_uBuffer];
s_uBuffer = (s_uBuffer + 1) % kTmpStrCount;

if (nDivisor != 1)
	{
	PANIC_IF((nDivisor == 0), ERROR_INVALID_PARAMETER);
	(void) STR_SPRINTF(szBuffer, kTmpStrLength, "%f", float(nValue) / nDivisor);
	}
else
	(void) ltoa(nValue, szBuffer, 10);
//
//	Use Win32 currency API
//
if (!::GetCurrencyFormat(NULL, NULL, szBuffer, NULL, pszResult, kTmpStrLength))
	{
	DEBUG_LOG1("::GetCurrencyFormat() failed with error #%lu\n", ::GetLastError());
	}

return pszResult;
}	/* end of StrCash() */
/****************************************************************************
**
**	StrDate() -- formats an absolute month, where 1 is Mar-01 into a 
**					mmm-yy format.
**
*/
StringPtr						// ptr to output string (buffer)
StrDate(
	uint16	wMonth)				// month 1..?
{
int32		nMonth,
			nYear;
StringPtr	pszResult = s_cBuffer[s_uBuffer];

s_uBuffer = (s_uBuffer + 1) % kTmpStrCount;

if (RCTFile::ConvertAbsMonth(wMonth, nMonth, nYear) &&
	nMonth >= 1 && nMonth <= 12)
	{
	const TextPtr	kMonth[] =	{ 
								"Jan", "Feb", "Mar", "Apr", "May", "Jun",
								"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
								};

	(void) STR_SPRINTF(pszResult, kTmpStrLength, 
						"%s-%02ld", 
						kMonth[nMonth-1], nYear);
	}
else
	STR_COPY(pszResult, "   -  ", kTmpStrLength);

return pszResult;
}	/* end of StrDate() */
/****************************************************************************
**
**	StrDate() --	formats a date into the format dd-mmm-yy
**
*/
StringPtr						// ptr to output string (buffer)
StrDate(
	int32		nDay,			// day (1..31)
	int32		nMonth,			// month (1..12)
	int32		nYear)			// year (1..99)
{
StringPtr		pszResult = s_cBuffer[s_uBuffer];

s_uBuffer = (s_uBuffer + 1) % kTmpStrCount;

if (nMonth >= 1 && nMonth <= 12)
	{
	const TextPtr	kMonth[] =	{ 
								"Jan", "Feb", "Mar", "Apr", "May", "Jun",
								"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
								};

	(void) STR_SPRINTF(pszResult, kTmpStrLength, 
						"%d-%s-%02ld", 
						nDay, kMonth[nMonth-1], nYear);
	}
else
	STR_COPY(pszResult, "  -   -  ", kTmpStrLength);

return pszResult;
}	/* end of StrDate() */
/****************************************************************************
**
**	StrNum() --	formats a binary long to a string with commas (replaces
**					ltoa() and printf()/sprintf() functions)
**
*/
StringPtr						// ptr to output string (buffer)
StrNum(
	int32	nValue,				// value to convert
	int32	nDivisor,			// divisor
	uint32	uDecimals)			// # of decimal places
{
StringPtr	pszResult;
Text		szBuffer[kTmpStrLength+1];

pszResult = s_cBuffer[s_uBuffer];
s_uBuffer = (s_uBuffer + 1) % kTmpStrCount;

if (nDivisor != 1)
	{
	PANIC_IF((nDivisor == 0), ERROR_INVALID_PARAMETER);
	(void) STR_SPRINTF(szBuffer, kTmpStrLength, "%f", float(nValue) / nDivisor);
	}
else
	(void) ltoa(nValue, szBuffer, 10);
//
//	Use Win32 currency API
//
NUMBERFMT	nf;

MEM_ZERO(&nf, sizeof(NUMBERFMT));
nf.NumDigits		= uDecimals;
nf.Grouping			= 3;
nf.lpDecimalSep		= ".";	//lint !e1776
nf.lpThousandSep	= ",";	//lint !e1776
nf.NegativeOrder	= 1;	// Negative sign, number. Example: -1.1
nf.LeadingZero		= 1;

if (!::GetNumberFormat(NULL, NULL, szBuffer, &nf, pszResult, kTmpStrLength))
	{
	DEBUG_LOG1("::GetNumberFormat() failed with error #%lu\n", ::GetLastError());
	}

return pszResult;
}	/* end of StrNum() */
/****************************************************************************
**
**	DebugLog() --	[DEBUG]
**
**	We use a custom DEBUG_LOG() macro so that we can automatically embed the 
**	source file and line.  This makes it easier to jump to the location of 
**	the error from the debugger.
**
*/
#ifdef	_DEBUG
void
DebugLog(
	TextPtr		pstrSrcFile,	// ptr to source file (__FILE__)
	uint32		uLine,			// source file line (__LINE__)
	TextPtr		pstrBase,		// ptr to output string
	...)						// [optional] printf()-style arguments
{
va_list		vaList;
Text		szOut[kMaxOutputWidth + 1];

(void) STR_SPRINTF(szOut, kMaxOutputWidth, "%s(%u): ", pstrSrcFile, uLine);
::OutputDebugString(szOut);

va_start(vaList, pstrBase);
(void) _vsnprintf(szOut, kMaxOutputWidth, pstrBase, vaList);
va_end(vaList);

::OutputDebugString(szOut);
}	/* end of DebugLog() */
#endif
/****************************************************************************
**
**	End of utility.cpp
**
****************************************************************************/
